<?php
// Initialize variables
 $ptitle = "Withdrawal Management";
 $adminId = 1; // Default admin ID (change as needed)
 $successMsg = '';
 $errorMsg = '';

// Include required files
require $_SERVER['DOCUMENT_ROOT']."/stream.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/includes/generalinclude.php";
// Fixed path - using relative path instead of absolute
require "includes/generalinclude.php";

// Set timezone
date_default_timezone_set('UTC');

// Create necessary tables if they don't exist
createTables();

function createTables() {
    global $link;
    
    // Create users table if not exists (using the existing users table)
    $tableCheck = $link->query("SHOW TABLES LIKE 'users'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE users (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            username VARCHAR(50) NOT NULL UNIQUE,
            email VARCHAR(100) NOT NULL UNIQUE,
            password VARCHAR(255) NOT NULL,
            funds DECIMAL(10,2) NOT NULL DEFAULT 0.00,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";
        $link->query($createTableQuery);
        
        // Insert default user
        $insertUserQuery = "INSERT INTO users (username, email, password, funds) VALUES ('admin', 'admin@example.com', '" . password_hash('admin123', PASSWORD_DEFAULT) . "', 5000.00)";
        $link->query($insertUserQuery);
    }
    
    // Create flow_admins table if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_admins'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE flow_admins (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            username VARCHAR(50) NOT NULL UNIQUE,
            email VARCHAR(100) NOT NULL UNIQUE,
            password VARCHAR(255) NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";
        $link->query($createTableQuery);
        
        // Insert default admin
        $insertAdminQuery = "INSERT INTO flow_admins (username, email, password) VALUES ('admin', 'admin@example.com', '" . password_hash('admin123', PASSWORD_DEFAULT) . "')";
        $link->query($insertAdminQuery);
    }
    
    // Create flow_settings table if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_settings'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE flow_settings (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            setting_key VARCHAR(50) NOT NULL UNIQUE,
            setting_value TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )";
        $link->query($createTableQuery);
        
        // Insert default settings
        $insertSettingsQuery = "INSERT INTO flow_settings (setting_key, setting_value) VALUES 
            ('withdrawal_enabled', '1'),
            ('min_withdrawal', '1000'),
            ('max_withdrawal', '1000000')";
        $link->query($insertSettingsQuery);
    }
    
    // Create flow_withdrawal_requests table if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_withdrawal_requests'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE flow_withdrawal_requests (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            user_id INT(11) NOT NULL,
            transaction_id VARCHAR(50) NOT NULL,
            amount DECIMAL(10,2) NOT NULL,
            method VARCHAR(50) NOT NULL,
            details TEXT NOT NULL,
            status ENUM('pending', 'approved', 'declined') NOT NULL DEFAULT 'pending',
            admin_notes TEXT NULL,
            approved_by INT(11) NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )";
        $link->query($createTableQuery);
    }
    
    // Create flow_transactions table if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_transactions'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE flow_transactions (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            user_id INT(11) NOT NULL,
            transaction_id VARCHAR(50) NOT NULL,
            amount DECIMAL(10,2) NOT NULL,
            type ENUM('deposit', 'withdrawal', 'transfer') NOT NULL,
            description TEXT NULL,
            created_at DATETIME NOT NULL
        )";
        $link->query($createTableQuery);
    }
}

// Helper function to update settings
function updateSetting($key, $value) {
    global $link;
    $checkQuery = $link->prepare("SELECT id FROM flow_settings WHERE setting_key = ?");
    $checkQuery->bind_param("s", $key);
    $checkQuery->execute();
    $result = $checkQuery->get_result();
    
    if ($result->num_rows > 0) {
        // Update existing setting
        $updateQuery = $link->prepare("UPDATE flow_settings SET setting_value = ? WHERE setting_key = ?");
        $updateQuery->bind_param("ss", $value, $key);
        $updateQuery->execute();
    } else {
        // Insert new setting
        $insertQuery = $link->prepare("INSERT INTO flow_settings (setting_key, setting_value) VALUES (?, ?)");
        $insertQuery->bind_param("ss", $key, $value);
        $insertQuery->execute();
    }
}

// Handle withdrawal toggle
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_withdrawals'])) {
    $withdrawalStatus = isset($_POST['withdrawal_status']) && $_POST['withdrawal_status'] == '1' ? 1 : 0;
    
    // Check if setting exists
    $checkQuery = $link->query("SELECT id FROM flow_settings WHERE setting_key = 'withdrawal_enabled'");
    $result = $checkQuery->fetch_assoc();
    
    if ($result) {
        // Update existing setting
        $updateQuery = $link->prepare("UPDATE flow_settings SET setting_value = ? WHERE setting_key = 'withdrawal_enabled'");
        $updateQuery->bind_param("i", $withdrawalStatus);
        $updateQuery->execute();
    } else {
        // Insert new setting
        $insertQuery = $link->prepare("INSERT INTO flow_settings (setting_key, setting_value) VALUES ('withdrawal_enabled', ?)");
        $insertQuery->bind_param("i", $withdrawalStatus);
        $insertQuery->execute();
    }
    
    $successMsg = "Withdrawal settings updated successfully!";
}

// Handle withdrawal limits update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_limits'])) {
    $minWithdrawal = $_POST['min_withdrawal'] ?? 0;
    $maxWithdrawal = $_POST['max_withdrawal'] ?? 0;
    
    // Validate: min must be less than or equal to max
    if ($minWithdrawal <= 0 || $maxWithdrawal <= 0) {
        $errorMsg = "Withdrawal limits must be greater than 0.";
    } elseif ($minWithdrawal > $maxWithdrawal) {
        $errorMsg = "Minimum withdrawal cannot be greater than maximum withdrawal.";
    } else {
        // Update min_withdrawal setting
        updateSetting('min_withdrawal', $minWithdrawal);
        // Update max_withdrawal setting
        updateSetting('max_withdrawal', $maxWithdrawal);
        
        $successMsg = "Withdrawal limits updated successfully!";
    }
}

// Handle withdrawal approval
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['approve_withdrawal'])) {
    $withdrawalId = $_POST['withdrawal_id'] ?? 0;
    $userId = 0;
    $amount = 0;
    
    if ($withdrawalId <= 0) {
        $errorMsg = "Invalid withdrawal ID";
    } else {
        // Get withdrawal details
        $withdrawalQuery = $link->prepare("SELECT * FROM flow_withdrawal_requests WHERE id = ? AND status = 'pending'");
        $withdrawalQuery->bind_param("i", $withdrawalId);
        $withdrawalQuery->execute();
        $withdrawalResult = $withdrawalQuery->get_result();
        
        if ($withdrawalResult->num_rows > 0) {
            $withdrawal = $withdrawalResult->fetch_assoc();
            $userId = $withdrawal['user_id'];
            $amount = $withdrawal['amount'];
            
            // Get user funds
            $userQuery = $link->prepare("SELECT funds FROM users WHERE id = ?");
            $userQuery->bind_param("i", $userId);
            $userQuery->execute();
            $userResult = $userQuery->get_result();
            
            if ($userResult->num_rows > 0) {
                $user = $userResult->fetch_assoc();
                $currentFunds = $user['funds'];
                
                // Check if user has sufficient funds
                if ($currentFunds >= $amount) {
                    // Start transaction
                    $link->begin_transaction();
                    
                    try {
                        // Deduct from user funds
                        $newFunds = $currentFunds - $amount;
                        
                        $updateFundsQuery = $link->prepare("UPDATE users SET funds = ? WHERE id = ?");
                        $updateFundsQuery->bind_param("di", $newFunds, $userId);
                        $updateFundsQuery->execute();
                        
                        // Update withdrawal status
                        $updateWithdrawalQuery = $link->prepare("UPDATE flow_withdrawal_requests SET status = 'approved', approved_by = ?, updated_at = NOW() WHERE id = ?");
                        $updateWithdrawalQuery->bind_param("ii", $adminId, $withdrawalId);
                        $updateWithdrawalQuery->execute();
                        
                        // Log the transaction
                        $logQuery = $link->prepare("INSERT INTO flow_transactions (user_id, transaction_id, amount, type, description, created_at) VALUES (?, ?, ?, 'withdrawal', ?, NOW())");
                        $transactionId = 'WD' . time() . rand(1000, 9999);
                        $description = "Withdrawal approved - Request ID: " . $withdrawalId;
                        $logQuery->bind_param("idss", $userId, $transactionId, $amount, $description);
                        $logQuery->execute();
                        
                        // Commit transaction
                        $link->commit();
                        
                        $successMsg = "Withdrawal approved successfully!";
                    } catch (Exception $e) {
                        // Rollback transaction on error
                        $link->rollback();
                        $errorMsg = "Error: " . $e->getMessage();
                    }
                } else {
                    $errorMsg = "Insufficient user funds for this withdrawal";
                }
            } else {
                $errorMsg = "User not found";
            }
        } else {
            $errorMsg = "Withdrawal not found or already processed";
        }
    }
}

// Handle withdrawal decline
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['decline_withdrawal'])) {
    $withdrawalId = $_POST['withdrawal_id'] ?? 0;
    $adminNotes = $_POST['admin_notes'] ?? '';
    
    if ($withdrawalId <= 0) {
        $errorMsg = "Invalid withdrawal ID";
    } else {
        // Update withdrawal status to declined
        $declineQuery = $link->prepare("UPDATE flow_withdrawal_requests SET status = 'declined', admin_notes = ?, approved_by = ?, updated_at = NOW() WHERE id = ? AND status = 'pending'");
        $declineQuery->bind_param("sii", $adminNotes, $adminId, $withdrawalId);
        
        if ($declineQuery->execute()) {
            if ($declineQuery->affected_rows > 0) {
                $successMsg = "Withdrawal has been declined successfully.";
            } else {
                $errorMsg = "Withdrawal not found or already processed.";
            }
        } else {
            $errorMsg = "Error declining withdrawal: " . $link->error;
        }
    }
}

// Get current withdrawal status
 $statusQuery = $link->query("SELECT setting_value FROM flow_settings WHERE setting_key = 'withdrawal_enabled'");
 $statusResult = $statusQuery->fetch_assoc();
 $withdrawalEnabled = $statusResult['setting_value'] == 1 ? true : false;

// Get minimum withdrawal amount
 $minWithdrawalQuery = $link->query("SELECT setting_value FROM flow_settings WHERE setting_key = 'min_withdrawal'");
 $minWithdrawalResult = $minWithdrawalQuery->fetch_assoc();
 $minWithdrawal = $minWithdrawalResult['setting_value'] ?? 1000;

// Get maximum withdrawal amount
 $maxWithdrawalQuery = $link->query("SELECT setting_value FROM flow_settings WHERE setting_key = 'max_withdrawal'");
 $maxWithdrawalResult = $maxWithdrawalQuery->fetch_assoc();
 $maxWithdrawal = $maxWithdrawalResult['setting_value'] ?? 1000000; // Default to 1,000,000 if not set

// Get all pending withdrawals
 $pendingWithdrawalsQuery = $link->prepare("SELECT w.*, u.username, u.email FROM flow_withdrawal_requests w JOIN users u ON w.user_id = u.id WHERE w.status = 'pending' ORDER BY w.created_at DESC");
 $pendingWithdrawalsQuery->execute();
 $pendingWithdrawalsResult = $pendingWithdrawalsQuery->get_result();

// Get all processed withdrawals
 $processedWithdrawalsQuery = $link->prepare("SELECT w.*, u.username, u.email FROM flow_withdrawal_requests w JOIN users u ON w.user_id = u.id WHERE w.status != 'pending' ORDER BY w.updated_at DESC LIMIT 50");
 $processedWithdrawalsQuery->execute();
 $processedWithdrawalsResult = $processedWithdrawalsQuery->get_result();

include "inc/header.php";
?>

<!-- Add Bootstrap JS for accordion functionality -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<style>
  :root {
    --primary-color: #4e73df;
    --primary-dark: #2e59d9;
    --primary-light: #f8f9fc;
    --text-color: #333;
    --bg-color: #f8f9fc;
    --card-bg: #ffffff;
    --card-border: #e3e6f0;
    --shadow: rgba(0, 0, 0, 0.1);
    --card-radius: 12px;
    --transition: all 0.3s ease;
    --success-color: #1cc88a;
    --warning-color: #f6c23e;
    --danger-color: #e74a3b;
  }

  * {
    box-sizing: border-box;
    margin: 0;
    padding: 0;
  }

  body {
    position: relative;
    margin: 0;
    padding: 0;
    overflow-x: hidden;
    color: var(--text-color);
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    background-color: var(--bg-color);
    font-size: 14px;
  }

  .nk-content {
    padding: 1rem 0;
  }

  .nk-block-head {
    margin-bottom: 1.5rem;
  }

  .nk-block-title {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--text-color);
    margin-bottom: 0.5rem;
  }

  .withdrawal-container {
    width: 100%;
    padding: 0 10px;
  }

  .withdrawal-section {
    background-color: var(--card-bg);
    color: var(--text-color);
    border-radius: var(--card-radius);
    box-shadow: 0 3px 10px var(--shadow);
    padding: 15px;
    border: 1px solid var(--card-border);
    transition: var(--transition);
    margin-bottom: 15px;
  }

  .section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-color);
    margin-bottom: 15px;
    padding-bottom: 8px;
    border-bottom: 1px solid var(--card-border);
    display: flex;
    align-items: center;
    gap: 8px;
  }

  .section-title i {
    color: var(--primary-color);
    font-size: 1rem;
  }

  /* Status Card with Toggle */
  .status-card {
    background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
    color: white;
    border-radius: var(--card-radius);
    padding: 15px;
    margin-bottom: 15px;
    box-shadow: 0 3px 10px var(--shadow);
    display: flex;
    justify-content: space-between;
    align-items: center;
  }

  .status-info {
    flex: 1;
  }

  .status-label {
    font-size: 0.85rem;
    opacity: 0.8;
    margin-bottom: 3px;
  }

  .status-value {
    font-size: 1.5rem;
    font-weight: 700;
  }

  .status-toggle {
    margin-left: 15px;
  }

  /* Toggle Switch */
  .toggle-switch {
    position: relative;
    display: inline-block;
    width: 50px;
    height: 24px;
  }

  .toggle-switch input {
    opacity: 0;
    width: 0;
    height: 0;
  }

  .toggle-slider {
    position: absolute;
    cursor: pointer;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(255, 255, 255, 0.3);
    transition: .4s;
    border-radius: 24px;
  }

  .toggle-slider:before {
    position: absolute;
    content: "";
    height: 16px;
    width: 16px;
    left: 4px;
    bottom: 4px;
    background-color: white;
    transition: .4s;
    border-radius: 50%;
  }

  input:checked + .toggle-slider {
    background-color: var(--success-color);
  }

  input:checked + .toggle-slider:before {
    transform: translateX(26px);
  }

  /* Form Styles */
  .form-group {
    margin-bottom: 12px;
  }

  .form-label {
    font-weight: 600;
    margin-bottom: 5px;
    color: var(--text-color);
    font-size: 0.9rem;
  }

  .form-control {
    border-radius: 6px;
    border: 1px solid var(--card-border);
    padding: 10px 12px;
    font-size: 0.9rem;
    transition: var(--transition);
    width: 100%;
  }

  .form-control:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 0.2rem rgba(78, 115, 223, 0.25);
    outline: none;
  }

  .btn {
    background-color: var(--primary-color);
    color: white;
    border: none;
    border-radius: 6px;
    padding: 10px 15px;
    font-size: 0.9rem;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition);
    box-shadow: 0 2px 5px var(--shadow);
    width: 100%;
  }

  .btn:hover {
    background-color: var(--primary-dark);
    transform: translateY(-1px);
    box-shadow: 0 3px 8px var(--shadow);
  }

  /* Accordion Styles */
  .accordion {
    border-radius: var(--card-radius);
    overflow: hidden;
    box-shadow: 0 3px 10px var(--shadow);
  }

  .accordion-item {
    border: none;
    margin-bottom: 5px;
    background-color: var(--card-bg);
  }

  .accordion-header {
    border-radius: var(--card-radius);
  }

  .accordion-button {
    background-color: var(--primary-color);
    color: white;
    font-weight: 600;
    border-radius: var(--card-radius);
    box-shadow: none;
    padding: 12px 15px;
    font-size: 0.9rem;
  }

  .accordion-button:not(.collapsed) {
    background-color: var(--primary-dark);
    color: white;
  }

  .accordion-button:focus {
    box-shadow: none;
    border-color: transparent;
  }

  .accordion-button::after {
    filter: brightness(0) invert(1);
    width: 1rem;
    height: 1rem;
    background-size: 1rem;
  }

  .accordion-body {
    padding: 15px;
    font-size: 0.85rem;
  }

  /* Detail Grid */
  .withdrawal-detail-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 10px;
    margin-bottom: 12px;
  }

  .withdrawal-detail-item {
    background-color: var(--primary-light);
    border-radius: 6px;
    padding: 10px;
  }

  .withdrawal-detail-label {
    font-weight: 600;
    color: #6c757d;
    font-size: 0.8rem;
    margin-bottom: 3px;
  }

  .withdrawal-detail-value {
    font-size: 0.9rem;
    font-weight: 600;
    color: var(--text-color);
    word-break: break-word;
  }

  /* Account Details Section */
  .account-details {
    margin-top: 12px;
  }

  .account-details h6 {
    font-size: 0.9rem;
    font-weight: 600;
    margin-bottom: 8px;
  }

  .account-details-content {
    background-color: var(--primary-light);
    border-radius: 6px;
    padding: 10px;
    font-size: 0.85rem;
    line-height: 1.4;
    max-height: 150px;
    overflow-y: auto;
    word-break: break-word;
    position: relative;
  }

  .copy-button {
    position: absolute;
    top: 10px;
    right: 10px;
    background-color: var(--primary-color);
    color: white;
    border: none;
    border-radius: 4px;
    padding: 5px 8px;
    font-size: 0.75rem;
    cursor: pointer;
    transition: var(--transition);
  }

  .copy-button:hover {
    background-color: var(--primary-dark);
  }

  /* Action Buttons */
  .action-buttons {
    display: flex;
    gap: 8px;
    margin-top: 15px;
  }

  .action-buttons .btn {
    flex: 1;
    padding: 8px 12px;
    font-size: 0.85rem;
  }

  .btn-approve {
    background-color: var(--success-color);
  }

  .btn-approve:hover {
    background-color: #17a673;
  }

  .btn-decline {
    background-color: var(--danger-color);
  }

  .btn-decline:hover {
    background-color: #d63384;
  }

  .btn-view {
    background-color: var(--warning-color);
  }

  .btn-view:hover {
    background-color: #e0a800;
  }

  /* Badges */
  .badge {
    font-size: 0.75rem;
    padding: 4px 8px;
    border-radius: 12px;
  }

  .badge-pending {
    background-color: var(--warning-color);
    color: white;
  }

  .badge-approved {
    background-color: var(--success-color);
    color: white;
  }

  .badge-declined {
    background-color: var(--danger-color);
    color: white;
  }

  /* Empty State */
  .no-withdrawal {
    text-align: center;
    font-size: 1rem;
    color: var(--text-color);
    padding: 20px;
    background-color: var(--primary-light);
    border-radius: var(--card-radius);
    margin: 15px 0;
    border: 1px dashed var(--primary-color);
  }

  /* Modal Styles */
  .modal {
    position: fixed;
    top: 0;
    left: 0;
    z-index: 1050;
    display: none;
    width: 100%;
    height: 100%;
    overflow: hidden;
    outline: 0;
    background-color: rgba(0, 0, 0, 0.5);
  }

  .modal.show {
    display: block;
  }

  .modal-dialog {
    position: relative;
    width: auto;
    margin: 10px;
    max-width: none;
  }

  .modal-content {
    position: relative;
    display: flex;
    flex-direction: column;
    width: 100%;
    pointer-events: auto;
    background-color: var(--card-bg);
    background-clip: padding-box;
    border: 1px solid var(--card-border);
    border-radius: var(--card-radius);
    outline: 0;
    box-shadow: 0 0.5rem 1rem rgba(0,0,0,.15);
    max-height: calc(100vh - 20px);
    overflow: hidden;
  }

  .modal-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 12px 15px;
    border-bottom: 1px solid var(--card-border);
  }

  .modal-title {
    margin-bottom: 0;
    line-height: 1.2;
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-color);
  }

  .btn-close {
    padding: 8px;
    margin: -8px -8px -8px auto;
    background-color: transparent;
    border: 0;
    font-size: 1.2rem;
    font-weight: 700;
    line-height: 1;
    color: #000;
    text-shadow: 0 1px 0 #fff;
    opacity: .5;
    cursor: pointer;
    border-radius: 50%;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
  }

  .btn-close:hover {
    color: #000;
    text-decoration: none;
    opacity: .75;
    background-color: rgba(0,0,0,.05);
  }

  .modal-body {
    position: relative;
    flex: 1 1 auto;
    padding: 15px;
    overflow-y: auto;
    max-height: calc(100vh - 120px);
  }

  .modal-footer {
    display: flex;
    flex-wrap: wrap;
    flex-shrink: 0;
    align-items: center;
    justify-content: flex-end;
    padding: 0.75rem;
    border-top: 1px solid var(--card-border);
    border-bottom-right-radius: calc(var(--card-radius) - 1px);
    border-bottom-left-radius: calc(var(--card-radius) - 1px);
  }

  .modal-footer > * {
    margin: 0.25rem;
  }

  /* Alert Styles */
  .alert {
    position: relative;
    padding: 12px 15px;
    margin-bottom: 12px;
    border: 1px solid transparent;
    border-radius: var(--card-radius);
    font-size: 0.85rem;
  }

  .alert-success {
    color: #fff;
    background-color: var(--success-color);
    border-color: var(--success-color);
  }

  .alert-danger {
    color: #fff;
    background-color: var(--danger-color);
    border-color: var(--danger-color);
  }

  /* Utility Classes */
  .text-center {
    text-align: center !important;
  }

  .bg-light {
    background-color: var(--primary-light) !important;
  }

  .rounded {
    border-radius: var(--card-radius) !important;
  }

  .p-2 {
    padding: 0.5rem !important;
  }

  .mt-2 {
    margin-top: 0.5rem !important;
  }

  .mb-2 {
    margin-bottom: 0.5rem !important;
  }

  .row {
    display: flex;
    flex-wrap: wrap;
    margin-right: -5px;
    margin-left: -5px;
  }

  .col-6 {
    flex: 0 0 50%;
    max-width: 50%;
    padding-right: 5px;
    padding-left: 5px;
  }

  .col-12 {
    flex: 0 0 100%;
    max-width: 100%;
    padding-right: 5px;
    padding-left: 5px;
  }

  /* Responsive adjustments */
  @media (min-width: 768px) {
    body {
      font-size: 16px;
    }
    
    .nk-content {
      padding: 1.5rem 0;
    }
    
    .nk-block-title {
      font-size: 1.8rem;
    }
    
    .withdrawal-container {
      padding: 0 15px;
    }
    
    .withdrawal-section {
      padding: 20px;
      margin-bottom: 20px;
    }
    
    .section-title {
      font-size: 1.3rem;
      margin-bottom: 20px;
    }
    
    .status-card {
      padding: 20px;
      margin-bottom: 20px;
    }
    
    .status-value {
      font-size: 1.8rem;
    }
    
    .form-label {
      font-size: 1rem;
    }
    
    .form-control {
      padding: 12px 15px;
      font-size: 1rem;
    }
    
    .btn {
      padding: 12px 20px;
      font-size: 1rem;
    }
    
    .accordion-button {
      padding: 15px 20px;
      font-size: 1rem;
    }
    
    .accordion-body {
      padding: 20px;
    }
    
    .withdrawal-detail-grid {
      grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
      gap: 15px;
    }
    
    .withdrawal-detail-item {
      padding: 15px;
    }
    
    .action-buttons .btn {
      padding: 10px 15px;
      font-size: 0.9rem;
    }
    
    .modal-dialog {
      margin: 1.75rem auto;
      max-width: 600px;
    }
    
    .modal-header {
      padding: 1rem 1rem;
    }
    
    .modal-body {
      padding: 1rem;
    }
    
    .modal-footer {
      padding: 1rem;
    }
  }
</style>

<div class="nk-content">
    <div class="container-fluid">
        <div class="nk-content-inner">
            <div class="nk-content-body">
                <div class="nk-block-head">
                    <div class="nk-block-head-between flex-wrap gap g-2">
                        <br>
                    </div>
                </div>
                
                <?php if (!empty($successMsg)): ?>
                <div class="alert alert-success">
                    <?php echo $successMsg; ?>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($errorMsg)): ?>
                <div class="alert alert-danger">
                    <?php echo $errorMsg; ?>
                </div>
                <?php endif; ?>
                
                <div class="withdrawal-container">
                    <!-- Section 1: Withdrawal Settings -->
                    <div class="withdrawal-section">
                        <h3 class="section-title">
                            <i class="fas fa-cog"></i>
                            Withdrawal Settings
                        </h3>
                        
                        <div class="status-card">
                            <div class="status-info">
                                <div class="status-label">Withdrawal Status</div>
                                <div class="status-value">
                                    <?php echo $withdrawalEnabled ? 'Enabled' : 'Disabled'; ?>
                                </div>
                            </div>
                            <div class="status-toggle">
                                <form method="POST" id="toggleForm" style="display: flex; align-items: center;">
                                    <input type="hidden" name="toggle_withdrawals" value="1">
                                    <input type="hidden" name="withdrawal_status" id="withdrawal_status" value="<?php echo $withdrawalEnabled ? '1' : '0'; ?>">
                                    <label class="toggle-switch">
                                        <input type="checkbox" id="toggleSwitch" <?php echo $withdrawalEnabled ? 'checked' : ''; ?> onchange="updateToggle()">
                                        <span class="toggle-slider"></span>
                                    </label>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Section 2: Withdrawal Limits -->
                    <div class="withdrawal-section">
                        <h3 class="section-title">
                            <i class="fas fa-sliders-h"></i>
                            Withdrawal Limits
                        </h3>
                        
                        <form method="POST" id="limitsForm">
                            <div class="row">
                                <div class="col-6">
                                    <div class="form-group">
                                        <label class="form-label">Minimum Withdrawal (₦)</label>
                                        <input type="number" class="form-control" name="min_withdrawal" value="<?php echo $minWithdrawal; ?>" step="0.01" min="0" required>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="form-group">
                                        <label class="form-label">Maximum Withdrawal (₦)</label>
                                        <input type="number" class="form-control" name="max_withdrawal" value="<?php echo $maxWithdrawal; ?>" step="0.01" min="0" required>
                                    </div>
                                </div>
                            </div>
                            <div class="form-group">
                                <button type="submit" name="update_limits" class="btn">Update Limits</button>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Section 3: Pending Withdrawals -->
                    <div class="withdrawal-section">
                        <h3 class="section-title">
                            <i class="fas fa-clock"></i>
                            Pending Withdrawals
                        </h3>
                        
                        <?php if ($pendingWithdrawalsResult->num_rows > 0): ?>
                        <div class="accordion" id="pendingWithdrawalsAccordion">
                            <?php 
                            $index = 0;
                            while ($withdrawal = $pendingWithdrawalsResult->fetch_assoc()): 
                                $id = $withdrawal['id'];
                                $username = $withdrawal['username'];
                                $email = $withdrawal['email'];
                                $amount = $withdrawal['amount'];
                                $method = $withdrawal['method'];
                                $details = $withdrawal['details'];
                                $created_at = date('M j, Y H:i', strtotime($withdrawal['created_at']));
                                
                                $statusClass = 'badge-pending';
                            ?>
                            <div class="accordion-item">
                                <h2 class="accordion-header" id="pendingHeading<?php echo $id ?>">
                                    <button class="accordion-button <?php echo $index > 0 ? 'collapsed' : '' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#pendingCollapse<?php echo $id ?>" aria-expanded="<?php echo $index == 0 ? 'true' : 'false' ?>" aria-controls="pendingCollapse<?php echo $id ?>">
                                        <div class="d-flex justify-content-between align-items-center w-100">
                                            <span><?php echo htmlspecialchars($username); ?> - ₦<?php echo number_format($amount, 2) ?></span>
                                            <span class="badge <?php echo $statusClass ?>">Pending</span>
                                        </div>
                                    </button>
                                </h2>
                                <div id="pendingCollapse<?php echo $id ?>" class="accordion-collapse collapse <?php echo $index == 0 ? 'show' : '' ?>" aria-labelledby="pendingHeading<?php echo $id ?>" data-bs-parent="#pendingWithdrawalsAccordion">
                                    <div class="accordion-body">
                                        <div class="withdrawal-detail-grid">
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Username</div>
                                                <div class="withdrawal-detail-value"><?php echo htmlspecialchars($username); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Email</div>
                                                <div class="withdrawal-detail-value"><?php echo htmlspecialchars($email); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Amount</div>
                                                <div class="withdrawal-detail-value">₦<?php echo number_format($amount, 2); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Method</div>
                                                <div class="withdrawal-detail-value"><?php echo htmlspecialchars($method); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Date</div>
                                                <div class="withdrawal-detail-value"><?php echo $created_at; ?></div>
                                            </div>
                                        </div>
                                        
                                        <div class="account-details">
                                            <h6>Account Details:</h6>
                                            <div class="account-details-content" id="account-details-<?php echo $id; ?>">
                                                <?php echo nl2br(htmlspecialchars($details)) ?>
                                                <button class="copy-button" onclick="copyAccountDetails(<?php echo $id; ?>)">
                                                    <i class="fas fa-copy"></i> Copy
                                                </button>
                                            </div>
                                        </div>
                                        
                                        <div class="action-buttons">
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="withdrawal_id" value="<?php echo $id; ?>">
                                                <input type="hidden" name="approve_withdrawal" value="1">
                                                <button type="submit" class="btn btn-approve">Approve</button>
                                            </form>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="withdrawal_id" value="<?php echo $id; ?>">
                                                <input type="hidden" name="decline_withdrawal" value="1">
                                                <button type="submit" class="btn btn-decline">Decline</button>
                                            </form>
                                            <button type="button" class="btn btn-view" data-bs-toggle="modal" data-bs-target="#accountDetailsModal<?php echo $id; ?>">View</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Account Details Modal -->
                            <div class="modal fade" id="accountDetailsModal<?php echo $id; ?>" tabindex="-1" aria-hidden="true">
                                <div class="modal-dialog">
                                    <div class="modal-content">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Account Details</h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                        </div>
                                        <div class="modal-body">
                                            <div class="row">
                                                <div class="col-6">
                                                    <div class="mb-2">
                                                        <div class="withdrawal-detail-label">Username</div>
                                                        <div class="withdrawal-detail-value"><?php echo htmlspecialchars($username); ?></div>
                                                    </div>
                                                </div>
                                                <div class="col-6">
                                                    <div class="mb-2">
                                                        <div class="withdrawal-detail-label">Email</div>
                                                        <div class="withdrawal-detail-value"><?php echo htmlspecialchars($email); ?></div>
                                                    </div>
                                                </div>
                                                <div class="col-6">
                                                    <div class="mb-2">
                                                        <div class="withdrawal-detail-label">Amount</div>
                                                        <div class="withdrawal-detail-value">₦<?php echo number_format($amount, 2); ?></div>
                                                    </div>
                                                </div>
                                                <div class="col-6">
                                                    <div class="mb-2">
                                                        <div class="withdrawal-detail-label">Method</div>
                                                        <div class="withdrawal-detail-value"><?php echo htmlspecialchars($method); ?></div>
                                                    </div>
                                                </div>
                                                <div class="col-12">
                                                    <div class="mb-2">
                                                        <div class="withdrawal-detail-label">Account Details</div>
                                                        <div class="account-details-content" id="modal-account-details-<?php echo $id; ?>">
                                                            <?php echo nl2br(htmlspecialchars($details)); ?>
                                                            <button class="copy-button" onclick="copyModalAccountDetails(<?php echo $id; ?>)">
                                                                <i class="fas fa-copy"></i> Copy
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php 
                            $index++;
                            endwhile; 
                            ?>
                        </div>
                        <?php else: ?>
                        <div class="no-withdrawal">No pending withdrawals found</div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Section 4: Processed Withdrawals -->
                    <div class="withdrawal-section">
                        <h3 class="section-title">
                            <i class="fas fa-history"></i>
                            Processed Withdrawals
                        </h3>
                        
                        <?php if ($processedWithdrawalsResult->num_rows > 0): ?>
                        <div class="accordion" id="processedWithdrawalsAccordion">
                            <?php 
                            $index = 0;
                            while ($withdrawal = $processedWithdrawalsResult->fetch_assoc()): 
                                $id = $withdrawal['id'];
                                $username = $withdrawal['username'];
                                $email = $withdrawal['email'];
                                $amount = $withdrawal['amount'];
                                $method = $withdrawal['method'];
                                $status = $withdrawal['status'];
                                $updated_at = date('M j, Y H:i', strtotime($withdrawal['updated_at']));
                                
                                $statusClass = '';
                                if ($status == 'approved') {
                                    $statusClass = 'badge-approved';
                                } else {
                                    $statusClass = 'badge-declined';
                                }
                            ?>
                            <div class="accordion-item">
                                <h2 class="accordion-header" id="processedHeading<?php echo $id ?>">
                                    <button class="accordion-button <?php echo $index > 0 ? 'collapsed' : '' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#processedCollapse<?php echo $id ?>" aria-expanded="<?php echo $index == 0 ? 'true' : 'false' ?>" aria-controls="processedCollapse<?php echo $id ?>">
                                        <div class="d-flex justify-content-between align-items-center w-100">
                                            <span><?php echo htmlspecialchars($username); ?> - ₦<?php echo number_format($amount, 2) ?></span>
                                            <span class="badge <?php echo $statusClass ?>"><?php echo ucfirst($status); ?></span>
                                        </div>
                                    </button>
                                </h2>
                                <div id="processedCollapse<?php echo $id ?>" class="accordion-collapse collapse <?php echo $index == 0 ? 'show' : '' ?>" aria-labelledby="processedHeading<?php echo $id ?>" data-bs-parent="#processedWithdrawalsAccordion">
                                    <div class="accordion-body">
                                        <div class="withdrawal-detail-grid">
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Username</div>
                                                <div class="withdrawal-detail-value"><?php echo htmlspecialchars($username); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Email</div>
                                                <div class="withdrawal-detail-value"><?php echo htmlspecialchars($email); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Amount</div>
                                                <div class="withdrawal-detail-value">₦<?php echo number_format($amount, 2); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Method</div>
                                                <div class="withdrawal-detail-value"><?php echo htmlspecialchars($method); ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Date</div>
                                                <div class="withdrawal-detail-value"><?php echo $updated_at; ?></div>
                                            </div>
                                            <div class="withdrawal-detail-item">
                                                <div class="withdrawal-detail-label">Status</div>
                                                <div class="withdrawal-detail-value"><span class="badge <?php echo $statusClass ?>"><?php echo ucfirst($status); ?></span></div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php 
                            $index++;
                            endwhile; 
                            ?>
                        </div>
                        <?php else: ?>
                        <div class="no-withdrawal">No processed withdrawals found</div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function updateToggle() {
    const toggleSwitch = document.getElementById('toggleSwitch');
    const hiddenInput = document.getElementById('withdrawal_status');
    hiddenInput.value = toggleSwitch.checked ? '1' : '0';
    document.getElementById('toggleForm').submit();
}

function copyAccountDetails(id) {
    const accountDetails = document.getElementById('account-details-' + id).innerText.replace(' Copy', '').trim();
    navigator.clipboard.writeText(accountDetails).then(() => {
        // Change button text temporarily to indicate success
        const copyButton = document.querySelector('#account-details-' + id + ' .copy-button');
        const originalText = copyButton.innerHTML;
        copyButton.innerHTML = '<i class="fas fa-check"></i> Copied!';
        copyButton.style.backgroundColor = 'var(--success-color)';
        
        // Reset button after 2 seconds
        setTimeout(() => {
            copyButton.innerHTML = originalText;
            copyButton.style.backgroundColor = 'var(--primary-color)';
        }, 2000);
    }).catch(err => {
        console.error('Failed to copy: ', err);
    });
}

function copyModalAccountDetails(id) {
    const accountDetails = document.getElementById('modal-account-details-' + id).innerText.replace(' Copy', '').trim();
    navigator.clipboard.writeText(accountDetails).then(() => {
        // Change button text temporarily to indicate success
        const copyButton = document.querySelector('#modal-account-details-' + id + ' .copy-button');
        const originalText = copyButton.innerHTML;
        copyButton.innerHTML = '<i class="fas fa-check"></i> Copied!';
        copyButton.style.backgroundColor = 'var(--success-color)';
        
        // Reset button after 2 seconds
        setTimeout(() => {
            copyButton.innerHTML = originalText;
            copyButton.style.backgroundColor = 'var(--primary-color)';
        }, 2000);
    }).catch(err => {
        console.error('Failed to copy: ', err);
    });
}
</script>

<?php include "includes/footer.php"; ?>