<?php
require $_SERVER['DOCUMENT_ROOT'] . "/stream.php";
require $_SERVER['DOCUMENT_ROOT'] . "$stream/includes/generalinclude.php";
require $_SERVER['DOCUMENT_ROOT'] . "$stream/user/includes/generalinclude.php";

 $ptitle = "Earn Tasks";
 $successMsg = '';
 $errorMsg = '';

// Set timezone
date_default_timezone_set('UTC');

// Get user ID from session
 $userId = $_SESSION['user_id'] ?? null;
 $username = $_SESSION['username'] ?? '';

// If user ID still not found, create a default user for testing
if (!$userId) {
    // Check if default user exists
    $checkUserQuery = $link->prepare("SELECT id FROM users WHERE username = 'Test User'");
    $checkUserQuery->execute();
    $checkResult = $checkUserQuery->get_result();
    
    if ($checkResult->num_rows == 0) {
        // Create default user
        $insertUserQuery = $link->prepare("INSERT INTO users (username, email, password, funds) VALUES (?, ?, ?, ?)");
        $defaultUsername = 'Test User';
        $defaultEmail = 'test@example.com';
        $defaultPassword = password_hash('password123', PASSWORD_DEFAULT);
        $defaultFunds = 0.00;
        $insertUserQuery->bind_param("sssd", $defaultUsername, $defaultEmail, $defaultPassword, $defaultFunds);
        $insertUserQuery->execute();
        $userId = $link->insert_id;
    } else {
        $userData = $checkResult->fetch_assoc();
        $userId = $userData['id'];
    }
    
    // Set user_id in session
    $_SESSION['user_id'] = $userId;
    $_SESSION['username'] = 'Test User';
    $_SESSION['email'] = 'test@example.com';
    $username = 'Test User';
}

// Get user details
 $userQuery = $link->prepare("SELECT id, username, email, funds FROM users WHERE id = ?");
 $userQuery->bind_param("i", $userId);
 $userQuery->execute();
 $userResult = $userQuery->get_result();

if ($userResult && $userResult->num_rows > 0) {
    $user = $userResult->fetch_assoc();
    $currentFunds = $user['funds'];
    $username = $user['username'];
    $email = $user['email'];
} else {
    $currentFunds = 0;
}

// Create necessary tables if they don't exist
createTables();

function createTables() {
    global $link;
    
    // Create flow_tasks table if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_tasks'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE flow_tasks (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL,
            link VARCHAR(255) NOT NULL,
            amount DECIMAL(10,2) NOT NULL,
            description TEXT NULL,
            status ENUM('active', 'inactive') NOT NULL DEFAULT 'active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )";
        $link->query($createTableQuery);
        
        // Insert some sample tasks
        $sampleTasks = [
            ['Like our Facebook page', 'https://facebook.com', 50.00, 'Like our official Facebook page and stay updated with our latest offers.'],
            ['Follow us on Instagram', 'https://instagram.com', 50.00, 'Follow our Instagram account for daily updates and exclusive content.'],
            ['Subscribe to our YouTube channel', 'https://youtube.com', 100.00, 'Subscribe to our YouTube channel and turn on notifications.'],
            ['Join our Telegram group', 'https://telegram.org', 75.00, 'Join our Telegram community to connect with other users.'],
            ['Share our website on social media', 'https://example.com', 100.00, 'Share our website on your social media accounts with the hashtag #Gainmart']
        ];
        
        foreach ($sampleTasks as $task) {
            $insertTask = $link->prepare("INSERT INTO flow_tasks (name, link, amount, description) VALUES (?, ?, ?, ?)");
            $insertTask->bind_param("ssds", $task[0], $task[1], $task[2], $task[3]);
            $insertTask->execute();
        }
    }
    
    // Create flow_user_tasks table if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_user_tasks'");
    if ($tableCheck->num_rows == 0) {
        $createTableQuery = "CREATE TABLE flow_user_tasks (
            id INT(11) AUTO_INCREMENT PRIMARY KEY,
            user_id INT(11) NOT NULL,
            task_id INT(11) NOT NULL,
            status ENUM('pending', 'completed') NOT NULL DEFAULT 'pending',
            reward_amount DECIMAL(10,2) NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            completed_at TIMESTAMP NULL,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (task_id) REFERENCES flow_tasks(id) ON DELETE CASCADE
        )";
        $link->query($createTableQuery);
    }
    
    // Update flow_transactions table to include task_reward type if not exists
    $tableCheck = $link->query("SHOW TABLES LIKE 'flow_transactions'");
    if ($tableCheck->num_rows > 0) {
        // Check if the table has the type column
        $columnCheck = $link->query("SHOW COLUMNS FROM flow_transactions LIKE 'type'");
        $columnInfo = $columnCheck->fetch_assoc();
        
        if ($columnInfo) {
            // Get the current ENUM values
            $typeInfo = $link->query("SHOW COLUMNS FROM flow_transactions WHERE Field = 'type'");
            $typeRow = $typeInfo->fetch_assoc();
            $type = $typeRow['Type'];
            
            // Extract ENUM values
            preg_match("/^enum\((.*)\)$/", $type, $matches);
            $enumValues = array();
            
            if (isset($matches[1])) {
                // Split the values and remove quotes
                $values = explode(",", $matches[1]);
                foreach ($values as $value) {
                    $enumValues[] = trim($value, "'");
                }
                
                // Check if 'task_reward' is already in the enum
                if (!in_array('task_reward', $enumValues)) {
                    // Add 'task_reward' to the enum values
                    $enumValues[] = 'task_reward';
                    
                    // Create a new temporary column with the updated ENUM
                    $tempColumn = 'type_temp';
                    $alterQuery1 = "ALTER TABLE flow_transactions ADD COLUMN {$tempColumn} ENUM('" . implode("', '", $enumValues) . "') NOT NULL";
                    $link->query($alterQuery1);
                    
                    // Copy data from old column to new column
                    $alterQuery2 = "UPDATE flow_transactions SET {$tempColumn} = type";
                    $link->query($alterQuery2);
                    
                    // Drop the old column
                    $alterQuery3 = "ALTER TABLE flow_transactions DROP COLUMN type";
                    $link->query($alterQuery3);
                    
                    // Rename the new column to the original name
                    $alterQuery4 = "ALTER TABLE flow_transactions CHANGE COLUMN {$tempColumn} type ENUM('" . implode("', '", $enumValues) . "') NOT NULL";
                    $link->query($alterQuery4);
                }
            }
        }
    }
}

// Handle task completion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['complete_task'])) {
    $taskId = $_POST['task_id'] ?? 0;
    
    if ($taskId <= 0) {
        $errorMsg = '<div class="alert alert-danger">Invalid task ID</div>';
    } else {
        // Check if user has already completed this task
        $checkQuery = $link->prepare("SELECT * FROM flow_user_tasks WHERE user_id = ? AND task_id = ? AND status = 'completed'");
        $checkQuery->bind_param("ii", $userId, $taskId);
        $checkQuery->execute();
        $checkResult = $checkQuery->get_result();
        
        if ($checkResult && $checkResult->num_rows > 0) {
            $errorMsg = '<div class="alert alert-danger">You have already completed this task</div>';
        } else {
            // Get task details
            $taskQuery = $link->prepare("SELECT * FROM flow_tasks WHERE id = ? AND status = 'active'");
            $taskQuery->bind_param("i", $taskId);
            $taskQuery->execute();
            $taskResult = $taskQuery->get_result();
            
            if ($taskResult && $taskResult->num_rows > 0) {
                $task = $taskResult->fetch_assoc();
                $rewardAmount = $task['amount'];
                
                // Start transaction
                $link->begin_transaction();
                
                try {
                    // Update user funds
                    $newFunds = $currentFunds + $rewardAmount;
                    $updateFundsQuery = $link->prepare("UPDATE users SET funds = ? WHERE id = ?");
                    $updateFundsQuery->bind_param("di", $newFunds, $userId);
                    $updateFundsQuery->execute();
                    
                    // Record task completion
                    $insertCompletionQuery = $link->prepare("INSERT INTO flow_user_tasks (user_id, task_id, status, reward_amount, completed_at) VALUES (?, ?, 'completed', ?, NOW())");
                    $insertCompletionQuery->bind_param("iid", $userId, $taskId, $rewardAmount);
                    $insertCompletionQuery->execute();
                    
                    // Record transaction
                    $transactionId = 'TASK' . time() . rand(1000, 9999);
                    $description = "Reward for task: " . $task['name'];
                    
                    // Check if the 'task_reward' type is supported
                    $typeCheck = $link->query("SHOW COLUMNS FROM flow_transactions LIKE 'type'");
                    $typeRow = $typeCheck->fetch_assoc();
                    $type = $typeRow['Type'];
                    
                    // Extract ENUM values
                    preg_match("/^enum\((.*)\)$/", $type, $matches);
                    $enumValues = array();
                    
                    if (isset($matches[1])) {
                        // Split the values and remove quotes
                        $values = explode(",", $matches[1]);
                        foreach ($values as $value) {
                            $enumValues[] = trim($value, "'");
                        }
                    }
                    
                    // Use 'task_reward' if available, otherwise use 'deposit' as fallback
                    $transactionType = in_array('task_reward', $enumValues) ? 'task_reward' : 'deposit';
                    
                    $insertTransactionQuery = $link->prepare("INSERT INTO flow_transactions (user_id, transaction_id, amount, type, description, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
                    $insertTransactionQuery->bind_param("idsss", $userId, $transactionId, $rewardAmount, $transactionType, $description);
                    $insertTransactionQuery->execute();
                    
                    // Commit transaction
                    $link->commit();
                    
                    $successMsg = '<div class="alert alert-success">Task completed successfully! ₦' . number_format($rewardAmount, 2) . ' has been added to your account.</div>';
                    $currentFunds = $newFunds; // Update current funds for display
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $link->rollback();
                    $errorMsg = '<div class="alert alert-danger">Error completing task: ' . $e->getMessage() . '</div>';
                }
            } else {
                $errorMsg = '<div class="alert alert-danger">Task not found or inactive</div>';
            }
        }
    }
}

// Get all active tasks that the user hasn't completed yet
 $tasksQuery = $link->prepare("SELECT t.* FROM flow_tasks t 
             LEFT JOIN flow_user_tasks ut ON t.id = ut.task_id AND ut.user_id = ? AND ut.status = 'completed'
             WHERE t.status = 'active' AND ut.id IS NULL
             ORDER BY t.created_at DESC");
 $tasksQuery->bind_param("i", $userId);
 $tasksQuery->execute();
 $tasksResult = $tasksQuery->get_result();
 $tasks = array();
if ($tasksResult) {
    while ($row = $tasksResult->fetch_assoc()) {
        $tasks[] = $row;
    }
}

// Get user's completed tasks
 $completedTasksQuery = $link->prepare("SELECT ut.*, t.name as task_name, t.link as task_link 
                     FROM flow_user_tasks ut 
                     JOIN flow_tasks t ON ut.task_id = t.id 
                     WHERE ut.user_id = ? AND ut.status = 'completed'
                     ORDER BY ut.completed_at DESC LIMIT 20");
 $completedTasksQuery->bind_param("i", $userId);
 $completedTasksQuery->execute();
 $completedTasksResult = $completedTasksQuery->get_result();
 $completedTasks = array();
if ($completedTasksResult) {
    while ($row = $completedTasksResult->fetch_assoc()) {
        $completedTasks[] = $row;
    }
}

// Calculate total earnings from tasks
 $earningsQuery = $link->prepare("SELECT SUM(reward_amount) as total FROM flow_user_tasks WHERE user_id = ? AND status = 'completed'");
 $earningsQuery->bind_param("i", $userId);
 $earningsQuery->execute();
 $earningsResult = $earningsQuery->get_result();
 $totalEarnings = 0;
if ($earningsResult) {
    $earnings = $earningsResult->fetch_assoc();
    $totalEarnings = $earnings['total'] ?? 0;
}

// Count completed tasks
 $countQuery = $link->prepare("SELECT COUNT(*) as count FROM flow_user_tasks WHERE user_id = ? AND status = 'completed'");
 $countQuery->bind_param("i", $userId);
 $countQuery->execute();
 $countResult = $countQuery->get_result();
 $completedCount = 0;
if ($countResult) {
    $count = $countResult->fetch_assoc();
    $completedCount = $count['count'] ?? 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <meta name="description" content="<?php echo $sitedesc ?>">
    <title><?php echo $ptitle ?> - <?php echo $sitename ?></title>
    <link rel="shortcut icon" href="../favicon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css"/>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        :root {
            /* Gainmart colors */
            --primary-blue: #75AFEE;
            --dark-blue: #4A8FE7;
            --darker-blue: #2E6CBF;
            --black: #000000;
            --dark-gray: #1A1A1A;
            --medium-gray: #2A2A2A;
            --light-gray: #3A3A3A;
            --text-light: #E0E0E0;
            --text-dim: #A0A0A0;
            --card-bg: #1F1F1F;
            --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            --input-bg: #2A2A2A;
            --border-color: #3A3A3A;
            --success: #4CAF50;
            --warning: #FFC107;
            --danger: #F44336;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--black);
            color: var(--text-light);
            transition: all 0.3s ease;
            min-height: 100vh;
            padding-bottom: 5rem; /* Added padding for bottom nav */
        }

        h1,h2,h3,h4,h5,h6 {
            font-weight: 700;
        }
        
        .container-fluid {
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
            margin-right: auto;
            margin-left: auto;
        }

        .row {
            display: flex;
            flex-wrap: wrap;
            margin-right: -15px;
            margin-left: -15px;
        }

        .col-12, .col-md-6 {
            position: relative;
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
        }

        @media (min-width: 768px) {
            .col-md-6 {
                flex: 0 0 50%;
                max-width: 50%;
            }
        }

        .g-gs {
            --bs-gutter-x: 1.5rem;
            --bs-gutter-y: 0;
            display: flex;
            flex-wrap: wrap;
            margin-top: calc(-1 * var(--bs-gutter-y));
            margin-right: calc(-.5 * var(--bs-gutter-x));
            margin-left: calc(-.5 * var(--bs-gutter-x));
        }

        .g-gs > * {
            flex-shrink: 0;
            width: 100%;
            max-width: 100%;
            padding-right: calc(var(--bs-gutter-x) * .5);
            padding-left: calc(var(--bs-gutter-x) * .5);
            margin-top: var(--bs-gutter-y);
        }

        .mt-3 {
            margin-top: 1rem !important;
        }

        .mt-4 {
            margin-top: 1.5rem !important;
        }

        .px-3 {
            padding-left: 1rem !important;
            padding-right: 1rem !important;
        }

        .mb-3 {
            margin-bottom: 1rem !important;
        }

        .mr-2 {
            margin-right: 0.5rem !important;
        }

        /* Header Styles */
        .header {
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            padding: 1rem 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .logo img {
            height: 40px;
        }

        .logo-text {
            font-size: 1.5rem;
            font-weight: 800;
            color: var(--primary-blue);
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .notification-btn {
            position: relative;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.2rem;
            cursor: pointer;
            padding: 0.75rem;
            border-radius: 50%;
            transition: all 0.3s ease;
        }

        .notification-btn:hover {
            background: var(--light-gray);
            color: var(--primary-blue);
        }

        .notification-badge {
            position: absolute;
            top: 0;
            right: 0;
            width: 10px;
            height: 10px;
            background: var(--danger);
            border-radius: 50%;
        }

        .account-section {
            position: relative;
        }

        .account-btn {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            cursor: pointer;
            padding: 0.5rem 1rem;
            border-radius: 2rem;
            transition: all 0.3s ease;
        }

        .account-btn:hover {
            background: var(--light-gray);
        }

        .account-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--primary-blue);
        }

        .account-info {
            display: flex;
            flex-direction: column;
            align-items: flex-start;
        }

        .account-name {
            font-weight: 600;
            font-size: 0.9rem;
        }

        .account-username {
            font-size: 0.8rem;
            color: var(--text-dim);
        }

        .account-dropdown {
            position: absolute;
            top: 100%;
            right: 0;
            margin-top: 0.5rem;
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            border-radius: 0.75rem;
            box-shadow: 0 10px 25px rgba(0,0,0,0.4);
            min-width: 220px;
            overflow: hidden;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.3s ease;
        }

        .account-dropdown.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }

        .dropdown-divider {
            height: 1px;
            background: var(--light-gray);
            margin: 0.5rem 0;
        }

        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: var(--text-light);
            text-decoration: none;
            transition: background-color 0.3s;
        }

        .dropdown-item:hover {
            background-color: var(--medium-gray);
        }

        .dropdown-item i {
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: var(--primary-blue);
        }

        /* Bottom Navigation Bar */
        .bottom-navigation {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: var(--dark-gray);
            box-shadow: 0 -2px 10px rgba(0,0,0,0.2);
            z-index: 100;
            padding: 0.75rem 0;
            display: flex;
            justify-content: space-around;
            align-items: center;
        }

        .bottom-nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
            color: var(--text-dim);
            text-decoration: none;
            width: 20%;
        }

        .bottom-nav-item:hover {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-item.active {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-icon {
            font-size: 1.5rem;
            margin-bottom: 0.25rem;
        }

        .bottom-nav-text {
            font-size: 0.7rem;
            font-weight: 600;
        }

        /* Right Sidebar Menu */
        .right-sidebar {
            position: fixed;
            top: 0;
            right: -50%; /* Hidden by default - half page width */
            width: 50%; /* Half page width */
            height: 100%;
            background: var(--dark-gray);
            z-index: 1001;
            transition: right 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
            overflow-y: auto;
            box-shadow: -5px 0 25px rgba(0,0,0,0.4);
        }

        .right-sidebar.active {
            right: 0;
        }

        .sidebar-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem;
            border-bottom: 1px solid var(--light-gray);
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
        }

        .sidebar-logo {
            height: 40px;
            filter: drop-shadow(0 2px 4px rgba(0,0,0,0.3));
        }

        .sidebar-close {
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
        }

        .sidebar-close:hover {
            background: var(--light-gray);
            transform: rotate(90deg);
        }

        .sidebar-content {
            padding: 1rem 0;
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
            min-height: calc(100% - 73px); /* Subtract header height */
        }

        .nk-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .nk-menu-item {
            margin: 0.25rem 0;
            opacity: 0;
            transform: translateX(20px);
            animation: slideInRight 0.5s forwards;
        }

        /* Stagger animation for menu items */
        .nk-menu-item:nth-child(1) { animation-delay: 0.1s; }
        .nk-menu-item:nth-child(2) { animation-delay: 0.15s; }
        .nk-menu-item:nth-child(3) { animation-delay: 0.2s; }
        .nk-menu-item:nth-child(4) { animation-delay: 0.25s; }
        .nk-menu-item:nth-child(5) { animation-delay: 0.3s; }
        .nk-menu-item:nth-child(6) { animation-delay: 0.35s; }
        .nk-menu-item:nth-child(7) { animation-delay: 0.4s; }
        .nk-menu-item:nth-child(8) { animation-delay: 0.45s; }
        .nk-menu-item:nth-child(9) { animation-delay: 0.5s; }
        .nk-menu-item:nth-child(10) { animation-delay: 0.55s; }
        .nk-menu-item:nth-child(11) { animation-delay: 0.6s; }
        .nk-menu-item:nth-child(12) { animation-delay: 0.65s; }
        .nk-menu-item:nth-child(13) { animation-delay: 0.7s; }
        .nk-menu-item:nth-child(14) { animation-delay: 0.75s; }
        .nk-menu-item:nth-child(15) { animation-delay: 0.8s; }

        @keyframes slideInRight {
            from {
                opacity: 0;
                transform: translateX(20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .nk-menu-link {
            display: flex;
            align-items: center;
            padding: 0.9rem 1.5rem;
            color: var(--text-light);
            text-decoration: none;
            transition: all 0.3s ease;
            border-radius: 0.75rem;
            margin: 0 0.75rem;
            position: relative;
            overflow: hidden;
        }

        .nk-menu-link::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary-blue);
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover {
            background: var(--medium-gray);
            transform: translateX(5px);
        }

        .nk-menu-link:hover::before {
            transform: scaleY(1);
        }

        .nk-menu-icon {
            margin-right: 1rem;
            color: var(--primary-blue);
            font-size: 1.2rem;
            width: 24px;
            text-align: center;
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover .nk-menu-icon {
            transform: scale(1.1);
        }

        .nk-menu-text {
            font-weight: 500;
            font-size: 1rem;
        }

        .nk-menu-heading {
            padding: 0.75rem 1.5rem;
            margin: 1.5rem 0 0.75rem;
            color: var(--text-dim);
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.1em;
            position: relative;
        }

        .nk-menu-heading::after {
            content: '';
            position: absolute;
            bottom: -0.5rem;
            left: 1.5rem;
            right: 1.5rem;
            height: 1px;
            background: var(--light-gray);
        }

        .overline-title {
            font-weight: 600;
        }

        .nk-menu-sub {
            list-style: none;
            padding: 0;
            margin: 0.25rem 0 0.25rem 2rem;
            background: var(--black);
            border-radius: 0.75rem;
            overflow: hidden;
            max-height: 0;
            opacity: 0;
            transition: all 0.4s ease;
        }

        .nk-menu-sub.active {
            max-height: 500px;
            opacity: 1;
        }

        .nk-menu-sub .nk-menu-item {
            margin: 0;
            opacity: 1;
            transform: none;
            animation: none;
        }

        .nk-menu-sub .nk-menu-link {
            padding: 0.7rem 1rem 0.7rem 1.5rem;
            font-size: 0.9rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
        }

        .nk-menu-sub .nk-menu-link::before {
            width: 3px;
        }

        .nk-menu-sub .nk-menu-icon {
            font-size: 1rem;
            margin-right: 0.75rem;
        }

        /* Menu toggle icon animation */
        .nk-menu-toggle .nk-menu-icon::after {
            content: '\f107';
            font-family: 'Font Awesome 6 Free';
            font-weight: 900;
            margin-left: auto;
            transition: transform 0.3s ease;
        }

        .nk-menu-toggle.active .nk-menu-icon::after {
            transform: rotate(180deg);
        }

        /* Overlay for sidebar */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        /* Page Title */
        .page-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.5rem;
            padding: 0 1rem;
        }

        /* Stats Cards */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 1rem;
            margin-bottom: 1.5rem;
        }

        .stat-card {
            background: var(--dark-gray);
            border-radius: 0.75rem;
            padding: 1.25rem;
            box-shadow: var(--card-shadow);
            transition: transform 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }

        .stat-title {
            font-size: 0.85rem;
            font-weight: 600;
            color: var(--text-dim);
        }

        .stat-icon {
            width: 36px;
            height: 36px;
            border-radius: 0.5rem;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
        }

        .stat-icon.primary {
            background: rgba(117, 175, 238, 0.15);
            color: var(--primary-blue);
        }

        .stat-icon.success {
            background: rgba(76, 175, 80, 0.15);
            color: var(--success);
        }

        .stat-icon.warning {
            background: rgba(255, 193, 7, 0.15);
            color: var(--warning);
        }

        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 0.25rem;
        }

        /* Tasks Section */
        .tasks-section {
            background: var(--dark-gray);
            border-radius: 0.75rem;
            padding: 1.25rem;
            margin-bottom: 1.5rem;
            box-shadow: var(--card-shadow);
        }

        .section-title {
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.25rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .section-title i {
            color: var(--primary-blue);
        }

        /* Task Card Styles */
        .task-card {
            background: var(--card-bg);
            border-radius: 0.75rem;
            padding: 1.25rem;
            margin-bottom: 1rem;
            box-shadow: var(--card-shadow);
            transition: transform 0.3s ease;
        }

        .task-card:hover {
            transform: translateY(-3px);
        }

        .task-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }

        .task-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-light);
        }

        .task-amount {
            font-size: 1rem;
            font-weight: 700;
            color: var(--success);
        }

        .task-description {
            color: var(--text-dim);
            margin-bottom: 1rem;
            font-size: 0.9rem;
        }

        .task-link {
            word-break: break-all;
            margin-bottom: 1rem;
            padding: 0.5rem;
            background-color: var(--input-bg);
            border-radius: 0.5rem;
            font-size: 0.85rem;
            color: var(--text-dim);
        }

        .task-actions {
            display: flex;
            gap: 0.75rem;
        }

        .task-btn {
            flex: 1;
            padding: 0.75rem;
            border-radius: 0.5rem;
            font-weight: 600;
            font-size: 0.9rem;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            text-align: center;
            text-decoration: none;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }

        .task-btn-primary {
            background: var(--primary-blue);
            color: var(--black);
        }

        .task-btn-primary:hover {
            background: var(--dark-blue);
        }

        .task-btn-success {
            background: var(--success);
            color: var(--black);
        }

        .task-btn-success:hover {
            background: #3d8b40;
        }

        .completed-task {
            opacity: 0.8;
            border-left: 4px solid var(--success);
        }

        /* Alert Styles */
        .alert {
            position: relative;
            padding: 0.75rem 1rem;
            margin-bottom: 1rem;
            border: 1px solid transparent;
            border-radius: 0.5rem;
            font-size: 0.9rem;
        }

        .alert-success {
            color: var(--black);
            background-color: var(--success);
            border-color: var(--success);
        }

        .alert-danger {
            color: var(--black);
            background-color: var(--danger);
            border-color: var(--danger);
        }

        .empty-state {
            text-align: center;
            padding: 2rem 1rem;
            color: var(--text-dim);
        }

        .empty-state-icon {
            font-size: 2.5rem;
            color: var(--light-gray);
            margin-bottom: 1rem;
        }

        .empty-state-text {
            font-size: 1rem;
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .header {
                padding: 0.75rem 1rem;
            }
            
            .logo-text {
                font-size: 1.2rem;
            }
            
            .account-info {
                display: none;
            }
            
            .right-sidebar {
                width: 80%; /* 80% width on mobile instead of 50% */
                right: -80%;
            }
            
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .task-actions {
                flex-direction: column;
            }
            
            .bottom-nav-text {
                font-size: 0.6rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="logo">
            <img src="/favicon.png" alt="Gainmart Logo">
            <div class="logo-text">Gainmart</div>
        </div>
        <div class="header-actions">
            <div class="account-section">
                <button class="account-btn" id="accountBtn">
                    <img src="../favicon.png" alt="Profile" class="account-avatar">
                    <div class="account-info">
                        <span class="account-name"><?php echo $username ?></span>
                        <span class="account-username">@<?php echo $username ?></span>
                    </div>
                    <i class="fas fa-chevron-down"></i>
                </button>
                
                <div class="account-dropdown" id="accountDropdown">
                    <a href="settings" class="dropdown-item">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                    <div class="dropdown-divider"></div>
                    <a href="logout" class="dropdown-item">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Log Out</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Right Sidebar Menu -->
    <div class="right-sidebar" id="rightSidebar">
        <div class="sidebar-header">
            <img src="favicon.png" alt="Logo" class="sidebar-logo">
            <button class="sidebar-close" id="sidebarClose">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="sidebar-content">
            <div class="nk-sidebar-menu" data-simplebar>
                <ul class="nk-menu">
                    <li class="nk-menu-item">
                        <a href="../user/index" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-home"></i></span>
                            <span class="nk-menu-text">Dashboard</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_packages.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-box"></i></span>
                            <span class="nk-menu-text">Investment Packages</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_withdrawal.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-money-bill-wave"></i></span>
                            <span class="nk-menu-text">Withdraw Funds</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="referrals" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-users"></i></span>
                            <span class="nk-menu-text">My Referrals</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="earn_tasks.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-tasks"></i></span>
                            <span class="nk-menu-text">Earn Tasks</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="settings" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-cog"></i></span>
                            <span class="nk-menu-text">Account Settings</span>
                        </a>
                    </li>
                    
                    <div class="mt-5"></div>
                </ul>
            </div>
        </div>
    </div>

    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <div class="nk-content px-0">
        <div class="container-fluid">
            <div class="nk-content-inner">
                <div class="nk-content-body">
                    <div class="row g-gs">

                        <div class="col-12 px-3 mt-4">
                            <h1 class="page-title">Earn Tasks</h1>
                        </div>

                        <!-- Stats Cards -->
                        <div class="col-12">
                            <div class="stats-container">
                                <div class="stat-card">
                                    <div class="stat-header">
                                        <div class="stat-title">Current Balance</div>
                                        <div class="stat-icon primary">
                                            <i class="fas fa-wallet"></i>
                                        </div>
                                    </div>
                                    <div class="stat-value">₦<?php echo number_format($currentFunds, 2) ?></div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-header">
                                        <div class="stat-title">Total Earnings</div>
                                        <div class="stat-icon success">
                                            <i class="fas fa-coins"></i>
                                        </div>
                                    </div>
                                    <div class="stat-value">₦<?php echo number_format($totalEarnings, 2) ?></div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-header">
                                        <div class="stat-title">Tasks Completed</div>
                                        <div class="stat-icon warning">
                                            <i class="fas fa-check-circle"></i>
                                        </div>
                                    </div>
                                    <div class="stat-value"><?php echo $completedCount ?></div>
                                </div>
                            </div>
                        </div>

                        <!-- Success/Error Messages -->
                        <div class="col-12">
                            <?php if (isset($successMsg)) echo $successMsg; ?>
                            <?php if (isset($errorMsg)) echo $errorMsg; ?>
                        </div>

                        <!-- Available Tasks Section -->
                        <div class="col-12">
                            <div class="tasks-section">
                                <div class="section-title">
                                    <i class="fas fa-tasks"></i>
                                    <span>Available Tasks</span>
                                </div>
                                
                                <?php if (count($tasks) > 0): ?>
                                    <?php foreach ($tasks as $task): ?>
                                        <div class="task-card">
                                            <div class="task-header">
                                                <div class="task-title"><?php echo htmlspecialchars($task['name']); ?></div>
                                                <div class="task-amount">₦<?php echo number_format($task['amount'], 2); ?></div>
                                            </div>
                                            
                                            <div class="task-description">
                                                <?php echo nl2br(htmlspecialchars($task['description'])); ?>
                                            </div>
                                            
                                            <div class="task-link">
                                                <strong>Link:</strong> <?php echo htmlspecialchars($task['link']); ?>
                                            </div>
                                            
                                            <div class="task-actions">
                                                <a href="<?php echo htmlspecialchars($task['link']); ?>" target="_blank" class="task-btn task-btn-primary">
                                                    <i class="fas fa-external-link-alt"></i>
                                                    <span>Open Task</span>
                                                </a>
                                                <form method="POST" style="display: flex; flex: 1;">
                                                    <input type="hidden" name="task_id" value="<?php echo $task['id']; ?>">
                                                    <input type="hidden" name="complete_task" value="1">
                                                    <button type="submit" class="task-btn task-btn-success">
                                                        <i class="fas fa-check"></i>
                                                        <span>Mark as Completed</span>
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="empty-state">
                                        <div class="empty-state-icon">
                                            <i class="fas fa-tasks"></i>
                                        </div>
                                        <div class="empty-state-text">No available tasks at the moment. Check back later!</div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Completed Tasks Section -->
                        <div class="col-12">
                            <div class="tasks-section">
                                <div class="section-title">
                                    <i class="fas fa-check-circle"></i>
                                    <span>Completed Tasks</span>
                                </div>
                                
                                <?php if (count($completedTasks) > 0): ?>
                                    <?php foreach ($completedTasks as $task): ?>
                                        <div class="task-card completed-task">
                                            <div class="task-header">
                                                <div class="task-title"><?php echo htmlspecialchars($task['task_name']); ?></div>
                                                <div class="task-amount">₦<?php echo number_format($task['reward_amount'], 2); ?></div>
                                            </div>
                                            
                                            <div class="task-link">
                                                <strong>Link:</strong> <?php echo htmlspecialchars($task['task_link']); ?>
                                            </div>
                                            
                                            <div class="text-dim">
                                                Completed on: <?php echo date('M j, Y H:i', strtotime($task['completed_at'])); ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="empty-state">
                                        <div class="empty-state-icon">
                                            <i class="fas fa-check-circle"></i>
                                        </div>
                                        <div class="empty-state-text">You haven't completed any tasks yet.</div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation Bar -->
    <div class="bottom-navigation">
        <a href="../user/index" class="bottom-nav-item">
            <i class="fas fa-home bottom-nav-icon"></i>
            <span class="bottom-nav-text">Dashboard</span>
        </a>
        <a href="user_packages.php" class="bottom-nav-item">
            <i class="fas fa-box bottom-nav-icon"></i>
            <span class="bottom-nav-text">Packages</span>
        </a>
        <a href="user_withdrawal.php" class="bottom-nav-item">
            <i class="fas fa-money-bill-wave bottom-nav-icon"></i>
            <span class="bottom-nav-text">Withdraw</span>
        </a>
        <a href="earn_tasks.php" class="bottom-nav-item active">
            <i class="fas fa-tasks bottom-nav-icon"></i>
            <span class="bottom-nav-text">Tasks</span>
        </a>
        <a href="settings" class="bottom-nav-item">
            <i class="fas fa-cog bottom-nav-icon"></i>
            <span class="bottom-nav-text">Settings</span>
        </a>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Account dropdown functionality
        const accountBtn = document.getElementById('accountBtn');
        const accountDropdown = document.getElementById('accountDropdown');
        
        if (accountBtn && accountDropdown) {
            accountBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                accountDropdown.classList.toggle('active');
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', function() {
                accountDropdown.classList.remove('active');
            });
            
            // Prevent dropdown from closing when clicking inside it
            accountDropdown.addEventListener('click', function(e) {
                e.stopPropagation();
            });
        }
        
        // Right sidebar functionality
        const hamburgerBtn = document.getElementById('hamburgerBtn');
        const rightSidebar = document.getElementById('rightSidebar');
        const sidebarClose = document.getElementById('sidebarClose');
        const sidebarOverlay = document.getElementById('sidebarOverlay');
        
        if (hamburgerBtn && rightSidebar && sidebarClose && sidebarOverlay) {
            // Open sidebar when hamburger is clicked
            hamburgerBtn.addEventListener('click', function() {
                rightSidebar.classList.add('active');
                sidebarOverlay.classList.add('active');
                document.body.style.overflow = 'hidden'; // Prevent body scroll when sidebar is open
                
                // Reset animations when opening
                const menuItems = rightSidebar.querySelectorAll('.nk-menu-item');
                menuItems.forEach(item => {
                    item.style.animation = 'none';
                    item.offsetHeight; // Trigger reflow
                    item.style.animation = null;
                });
            });
            
            // Close sidebar when close button is clicked
            sidebarClose.addEventListener('click', function() {
                rightSidebar.classList.remove('active');
                sidebarOverlay.classList.remove('active');
                document.body.style.overflow = ''; // Restore body scroll
            });
            
            // Close sidebar when overlay is clicked
            sidebarOverlay.addEventListener('click', function() {
                rightSidebar.classList.remove('active');
                sidebarOverlay.classList.remove('active');
                document.body.style.overflow = ''; // Restore body scroll
            });
            
            // Handle submenu toggles
            const menuToggles = document.querySelectorAll('.nk-menu-toggle');
            menuToggles.forEach(toggle => {
                toggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    const parentItem = this.parentElement;
                    const submenu = parentItem.querySelector('.nk-menu-sub');
                    
                    if (submenu) {
                        submenu.classList.toggle('active');
                        this.classList.toggle('active');
                    }
                });
            });
        }
    });
    </script>
</body>
</html>