<?php
// user_deposit.php
require $_SERVER['DOCUMENT_ROOT']."/stream.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/includes/generalinclude.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/user/includes/generalinclude.php";

// Set timezone
date_default_timezone_set('UTC');

// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Get user ID from session or database
 $userId = null;
 $username = null;

// First, try to get user ID from session
if (isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
} 
// If not in session, try to get from username in session
elseif (isset($_SESSION['username'])) {
    $username = $_SESSION['username'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE username = ?");
    $getUserQuery->bind_param("s", $username);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}
// If still not found, try to get from email in session
elseif (isset($_SESSION['email'])) {
    $email = $_SESSION['email'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE email = ?");
    $getUserQuery->bind_param("s", $email);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}

// If user ID still not found, create a default user for testing
if (!$userId) {
    // Check if default user exists
    $checkUserQuery = $link->prepare("SELECT id FROM users WHERE username = 'Test User'");
    $checkUserQuery->execute();
    $checkResult = $checkUserQuery->get_result();
    
    if ($checkResult->num_rows == 0) {
        // Create default user
        $insertUserQuery = $link->prepare("INSERT INTO users (username, email, password, funds) VALUES (?, ?, ?, ?)");
        $defaultUsername = 'Test User';
        $defaultEmail = 'test@example.com';
        $defaultPassword = password_hash('password123', PASSWORD_DEFAULT);
        $defaultFunds = 0.00;
        $insertUserQuery->bind_param("sssd", $defaultUsername, $defaultEmail, $defaultPassword, $defaultFunds);
        $insertUserQuery->execute();
        $userId = $link->insert_id;
    } else {
        $userData = $checkResult->fetch_assoc();
        $userId = $userData['id'];
    }
    
    // Set user_id in session
    $_SESSION['user_id'] = $userId;
    $_SESSION['username'] = 'Test User';
    $_SESSION['email'] = 'test@example.com';
}

 $successMsg = '';
 $errorMsg = '';

// Get user's current funds
 $getUserFunds = $link->prepare("SELECT funds FROM users WHERE id = ?");
 $getUserFunds->bind_param("i", $userId);
 $getUserFunds->execute();
 $userFundsResult = $getUserFunds->get_result();
 $userFundsRow = $userFundsResult->fetch_assoc();
 $currentFunds = $userFundsRow ? $userFundsRow['funds'] : 0;

// Get bank settings
 $bankSettings = $link->query("SELECT * FROM flow_deposit_settings WHERE payment_method = 'Bank Transfer' AND is_active = 1 LIMIT 1");
 $bankData = $bankSettings->fetch_assoc();

// Handle deposit submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_deposit'])) {
    $amount = $_POST['amount'] ?? 0;
    
    // Validate amount
    if ($amount <= 0) {
        $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Please enter a valid amount</div>';
    } elseif ($bankData && ($amount < $bankData['min_amount'] || $amount > $bankData['max_amount'])) {
        $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Amount must be between ₦' . number_format($bankData['min_amount'], 2) . ' and ₦' . number_format($bankData['max_amount'], 2) . '</div>';
    } else {
        // Handle file upload
        if (isset($_FILES['proof_image']) && $_FILES['proof_image']['error'] === UPLOAD_ERR_OK) {
            $fileTmpPath = $_FILES['proof_image']['tmp_name'];
            $fileName = $_FILES['proof_image']['name'];
            $fileSize = $_FILES['proof_image']['size'];
            $fileType = $_FILES['proof_image']['type'];
            $fileNameCmps = explode(".", $fileName);
            $fileExtension = strtolower(end($fileNameCmps));
            
            // Check file type
            $allowedfileExtensions = array('jpg', 'jpeg', 'png', 'gif');
            if (in_array($fileExtension, $allowedfileExtensions)) {
                // Create upload directory if it doesn't exist
                $uploadDir = $_SERVER['DOCUMENT_ROOT'] . "/stream/uploads/deposits/";
                
                // Check if directory exists and is writable
                if (!file_exists($uploadDir)) {
                    // Try to create directory with recursive flag
                    if (!mkdir($uploadDir, 0755, true)) {
                        $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Failed to create upload directory. Please contact administrator.</div>';
                    } else {
                        // Set proper permissions after creation
                        chmod($uploadDir, 0755);
                    }
                }
                
                // Check if directory is writable
                if (empty($errorMsg) && !is_writable($uploadDir)) {
                    // Try to fix permissions
                    if (!chmod($uploadDir, 0755)) {
                        $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Upload directory is not writable. Please contact administrator.</div>';
                    }
                }
                
                if (empty($errorMsg)) {
                    // Generate unique filename
                    $newFileName = md5(time() . $userId) . '.' . $fileExtension;
                    $uploadPath = $uploadDir . $newFileName;
                    
                    // Move file to upload directory
                    if (move_uploaded_file($fileTmpPath, $uploadPath)) {
                        // Generate transaction ID
                        $transactionId = 'TXN' . time() . rand(1000, 9999);
                        
                        // Insert deposit record
                        $insertDeposit = $link->prepare("INSERT INTO flow_deposits (user_id, transaction_id, amount, payment_method, proof_image, status) VALUES (?, ?, ?, 'Bank Transfer', ?, 'pending')");
                        $insertDeposit->bind_param("isds", $userId, $transactionId, $amount, $newFileName);
                        
                        if ($insertDeposit->execute()) {
                            $successMsg = '<div class="copy-notification">Deposit submitted successfully! Your deposit is now pending approval.</div>';
                        } else {
                            $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Error submitting deposit. Please try again.</div>';
                        }
                    } else {
                        $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Error uploading file. Please try again.</div>';
                    }
                }
            } else {
                $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Invalid file type. Only JPG, JPEG, PNG, and GIF files are allowed.</div>';
            }
        } else {
            // Get specific error message
            $uploadError = $_FILES['proof_image']['error'];
            switch ($uploadError) {
                case UPLOAD_ERR_INI_SIZE:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">The uploaded file exceeds the upload_max_filesize directive in php.ini.</div>';
                    break;
                case UPLOAD_ERR_FORM_SIZE:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form.</div>';
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">The uploaded file was only partially uploaded.</div>';
                    break;
                case UPLOAD_ERR_NO_FILE:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">No file was uploaded.</div>';
                    break;
                case UPLOAD_ERR_NO_TMP_DIR:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Missing a temporary folder.</div>';
                    break;
                case UPLOAD_ERR_CANT_WRITE:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Failed to write file to disk.</div>';
                    break;
                case UPLOAD_ERR_EXTENSION:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">File upload stopped by extension.</div>';
                    break;
                default:
                    $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Please upload a proof of payment image.</div>';
                    break;
            }
        }
    }
}

// Get user deposits
 $getDeposits = $link->prepare("SELECT * FROM flow_deposits WHERE user_id = ? ORDER BY created_at DESC");
 $getDeposits->bind_param("i", $userId);
 $getDeposits->execute();
 $depositsResult = $getDeposits->get_result();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <meta name="description" content="<?php echo $sitedesc ?>">
    <title>Deposit Funds - <?php echo $sitename ?></title>
    <link rel="shortcut icon" href="../favicon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css"/>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        :root {
            /* Gainmart colors */
            --primary-blue: #75AFEE;
            --dark-blue: #4A8FE7;
            --darker-blue: #2E6CBF;
            --black: #000000;
            --dark-gray: #1A1A1A;
            --medium-gray: #2A2A2A;
            --light-gray: #3A3A3A;
            --text-light: #E0E0E0;
            --text-dim: #A0A0A0;
            --card-bg: #1F1F1F;
            --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            --input-bg: #2A2A2A;
            --border-color: #3A3A3A;
            --success: #4CAF50;
            --warning: #FFC107;
            --danger: #F44336;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--black);
            color: var(--text-light);
            transition: all 0.3s ease;
            min-height: 100vh;
            padding-bottom: 5rem; /* Added padding for bottom nav */
        }

        h1,h2,h3,h4,h5,h6 {
            font-weight: 700;
        }
        
        .container-fluid {
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
            margin-right: auto;
            margin-left: auto;
        }

        .row {
            display: flex;
            flex-wrap: wrap;
            margin-right: -15px;
            margin-left: -15px;
        }

        .col-12, .col-md-6 {
            position: relative;
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
        }

        @media (min-width: 768px) {
            .col-md-6 {
                flex: 0 0 50%;
                max-width: 50%;
            }
        }

        .g-gs {
            --bs-gutter-x: 1.5rem;
            --bs-gutter-y: 0;
            display: flex;
            flex-wrap: wrap;
            margin-top: calc(-1 * var(--bs-gutter-y));
            margin-right: calc(-.5 * var(--bs-gutter-x));
            margin-left: calc(-.5 * var(--bs-gutter-x));
        }

        .g-gs > * {
            flex-shrink: 0;
            width: 100%;
            max-width: 100%;
            padding-right: calc(var(--bs-gutter-x) * .5);
            padding-left: calc(var(--bs-gutter-x) * .5);
            margin-top: var(--bs-gutter-y);
        }

        .mt-3 {
            margin-top: 1rem !important;
        }

        .mt-4 {
            margin-top: 1.5rem !important;
        }

        .px-3 {
            padding-left: 1rem !important;
            padding-right: 1rem !important;
        }

        .mb-3 {
            margin-bottom: 1rem !important;
        }

        .mb-4 {
            margin-bottom: 1.5rem !important;
        }

        .mr-2 {
            margin-right: 0.5rem !important;
        }

        /* Header Styles */
        .header {
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            padding: 1rem 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .logo img {
            height: 40px;
        }

        .logo-text {
            font-size: 1.5rem;
            font-weight: 800;
            color: var(--primary-blue);
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .notification-btn {
            position: relative;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.2rem;
            cursor: pointer;
            padding: 0.75rem;
            border-radius: 50%;
            transition: all 0.3s ease;
        }

        .notification-btn:hover {
            background: var(--light-gray);
            color: var(--primary-blue);
        }

        .notification-badge {
            position: absolute;
            top: 0;
            right: 0;
            width: 10px;
            height: 10px;
            background: var(--danger);
            border-radius: 50%;
        }

        .account-section {
            position: relative;
        }

        .account-btn {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            cursor: pointer;
            padding: 0.5rem 1rem;
            border-radius: 2rem;
            transition: all 0.3s ease;
        }

        .account-btn:hover {
            background: var(--light-gray);
        }

        .account-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--primary-blue);
        }

        .account-info {
            display: flex;
            flex-direction: column;
            align-items: flex-start;
        }

        .account-name {
            font-weight: 600;
            font-size: 0.9rem;
        }

        .account-username {
            font-size: 0.8rem;
            color: var(--text-dim);
        }

        .account-dropdown {
            position: absolute;
            top: 100%;
            right: 0;
            margin-top: 0.5rem;
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            border-radius: 0.75rem;
            box-shadow: 0 10px 25px rgba(0,0,0,0.4);
            min-width: 220px;
            overflow: hidden;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.3s ease;
        }

        .account-dropdown.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }

        .dropdown-divider {
            height: 1px;
            background: var(--light-gray);
            margin: 0.5rem 0;
        }

        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: var(--text-light);
            text-decoration: none;
            transition: background-color 0.3s;
        }

        .dropdown-item:hover {
            background-color: var(--medium-gray);
        }

        .dropdown-item i {
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: var(--primary-blue);
        }

        /* Bottom Navigation Bar */
        .bottom-navigation {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: var(--dark-gray);
            box-shadow: 0 -2px 10px rgba(0,0,0,0.2);
            z-index: 100;
            padding: 0.75rem 0;
            display: flex;
            justify-content: space-around;
            align-items: center;
        }

        .bottom-nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
            color: var(--text-dim);
            text-decoration: none;
            width: 20%;
        }

        .bottom-nav-item:hover {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-item.active {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-icon {
            font-size: 1.5rem;
            margin-bottom: 0.25rem;
        }

        .bottom-nav-text {
            font-size: 0.7rem;
            font-weight: 600;
        }

        /* Right Sidebar Menu */
        .right-sidebar {
            position: fixed;
            top: 0;
            right: -50%; /* Hidden by default - half page width */
            width: 50%; /* Half page width */
            height: 100%;
            background: var(--dark-gray);
            z-index: 1001;
            transition: right 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
            overflow-y: auto;
            box-shadow: -5px 0 25px rgba(0,0,0,0.4);
        }

        .right-sidebar.active {
            right: 0;
        }

        .sidebar-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem;
            border-bottom: 1px solid var(--light-gray);
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
        }

        .sidebar-logo {
            height: 40px;
            filter: drop-shadow(0 2px 4px rgba(0,0,0,0.3));
        }

        .sidebar-close {
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
        }

        .sidebar-close:hover {
            background: var(--light-gray);
            transform: rotate(90deg);
        }

        .sidebar-content {
            padding: 1rem 0;
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
            min-height: calc(100% - 73px); /* Subtract header height */
        }

        .nk-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .nk-menu-item {
            margin: 0.25rem 0;
            opacity: 0;
            transform: translateX(20px);
            animation: slideInRight 0.5s forwards;
        }

        /* Stagger animation for menu items */
        .nk-menu-item:nth-child(1) { animation-delay: 0.1s; }
        .nk-menu-item:nth-child(2) { animation-delay: 0.15s; }
        .nk-menu-item:nth-child(3) { animation-delay: 0.2s; }
        .nk-menu-item:nth-child(4) { animation-delay: 0.25s; }
        .nk-menu-item:nth-child(5) { animation-delay: 0.3s; }
        .nk-menu-item:nth-child(6) { animation-delay: 0.35s; }
        .nk-menu-item:nth-child(7) { animation-delay: 0.4s; }
        .nk-menu-item:nth-child(8) { animation-delay: 0.45s; }
        .nk-menu-item:nth-child(9) { animation-delay: 0.5s; }
        .nk-menu-item:nth-child(10) { animation-delay: 0.55s; }
        .nk-menu-item:nth-child(11) { animation-delay: 0.6s; }
        .nk-menu-item:nth-child(12) { animation-delay: 0.65s; }
        .nk-menu-item:nth-child(13) { animation-delay: 0.7s; }
        .nk-menu-item:nth-child(14) { animation-delay: 0.75s; }
        .nk-menu-item:nth-child(15) { animation-delay: 0.8s; }

        @keyframes slideInRight {
            from {
                opacity: 0;
                transform: translateX(20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .nk-menu-link {
            display: flex;
            align-items: center;
            padding: 0.9rem 1.5rem;
            color: var(--text-light);
            text-decoration: none;
            transition: all 0.3s ease;
            border-radius: 0.75rem;
            margin: 0 0.75rem;
            position: relative;
            overflow: hidden;
        }

        .nk-menu-link::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary-blue);
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover {
            background: var(--medium-gray);
            transform: translateX(5px);
        }

        .nk-menu-link:hover::before {
            transform: scaleY(1);
        }

        .nk-menu-icon {
            margin-right: 1rem;
            color: var(--primary-blue);
            font-size: 1.2rem;
            width: 24px;
            text-align: center;
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover .nk-menu-icon {
            transform: scale(1.1);
        }

        .nk-menu-text {
            font-weight: 500;
            font-size: 1rem;
        }

        .nk-menu-heading {
            padding: 0.75rem 1.5rem;
            margin: 1.5rem 0 0.75rem;
            color: var(--text-dim);
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.1em;
            position: relative;
        }

        .nk-menu-heading::after {
            content: '';
            position: absolute;
            bottom: -0.5rem;
            left: 1.5rem;
            right: 1.5rem;
            height: 1px;
            background: var(--light-gray);
        }

        .overline-title {
            font-weight: 600;
        }

        .nk-menu-sub {
            list-style: none;
            padding: 0;
            margin: 0.25rem 0 0.25rem 2rem;
            background: var(--black);
            border-radius: 0.75rem;
            overflow: hidden;
            max-height: 0;
            opacity: 0;
            transition: all 0.4s ease;
        }

        .nk-menu-sub.active {
            max-height: 500px;
            opacity: 1;
        }

        .nk-menu-sub .nk-menu-item {
            margin: 0;
            opacity: 1;
            transform: none;
            animation: none;
        }

        .nk-menu-sub .nk-menu-link {
            padding: 0.7rem 1rem 0.7rem 1.5rem;
            font-size: 0.9rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
        }

        .nk-menu-sub .nk-menu-link::before {
            width: 3px;
        }

        .nk-menu-sub .nk-menu-icon {
            font-size: 1rem;
            margin-right: 0.75rem;
        }

        /* Menu toggle icon animation */
        .nk-menu-toggle .nk-menu-icon::after {
            content: '\f107';
            font-family: 'Font Awesome 6 Free';
            font-weight: 900;
            margin-left: auto;
            transition: transform 0.3s ease;
        }

        .nk-menu-toggle.active .nk-menu-icon::after {
            transform: rotate(180deg);
        }

        /* Overlay for sidebar */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        /* Page Title */
        .page-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.5rem;
            padding: 0 1rem;
        }

        /* Balance Card */
        .balance-card {
            background: linear-gradient(135deg, var(--dark-gray) 0%, var(--medium-gray) 100%);
            border-radius: 1rem;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            box-shadow: var(--card-shadow);
            position: relative;
            overflow: hidden;
        }

        .balance-card::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 150px;
            height: 150px;
            background: var(--primary-blue);
            opacity: 0.1;
            border-radius: 50%;
            transform: translate(50px, -50px);
        }

        .balance-card::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100px;
            height: 100px;
            background: var(--primary-blue);
            opacity: 0.1;
            border-radius: 50%;
            transform: translate(-30px, 30px);
        }

        .balance-label {
            font-size: 0.9rem;
            font-weight: 600;
            color: var(--text-dim);
            margin-bottom: 0.5rem;
            position: relative;
            z-index: 1;
        }

        .balance-amount {
            font-size: 2.5rem;
            font-weight: 800;
            color: var(--primary-blue);
            margin-bottom: 1rem;
            position: relative;
            z-index: 1;
        }

        /* Bank Details Card */
        .bank-details-card {
            background: var(--dark-gray);
            border-radius: 1rem;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            box-shadow: var(--card-shadow);
        }

        .section-title {
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .section-title i {
            color: var(--primary-blue);
        }

        .bank-info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.75rem 0;
            border-bottom: 1px solid var(--border-color);
        }

        .bank-info-row:last-child {
            border-bottom: none;
        }

        .bank-info-label {
            font-weight: 600;
            color: var(--text-dim);
            font-size: 0.9rem;
        }

        .bank-info-value {
            font-weight: 600;
            color: var(--text-light);
            font-size: 1rem;
            display: flex;
            align-items: center;
        }

        .copy-button {
            margin-left: 0.75rem;
            background: var(--primary-blue);
            color: var(--black);
            border: none;
            border-radius: 0.5rem;
            padding: 0.5rem 0.75rem;
            font-size: 0.75rem;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .copy-button:hover {
            background: var(--dark-blue);
            transform: translateY(-2px);
        }

        .amount-limits {
            display: flex;
            justify-content: space-between;
            margin-top: 1.25rem;
            gap: 1rem;
        }

        .amount-limit {
            text-align: center;
            background: var(--medium-gray);
            border-radius: 0.75rem;
            padding: 1rem;
            flex: 1;
        }

        .amount-limit-label {
            font-size: 0.8rem;
            color: var(--text-dim);
            margin-bottom: 0.5rem;
        }

        .amount-limit-value {
            font-size: 1.1rem;
            font-weight: 700;
            color: var(--primary-blue);
        }

        /* Deposit Form Card */
        .deposit-form-card {
            background: var(--dark-gray);
            border-radius: 1rem;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            box-shadow: var(--card-shadow);
        }

        .form-group {
            margin-bottom: 1.25rem;
        }

        .form-label {
            display: block;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--text-light);
            font-size: 0.9rem;
        }

        .form-control {
            border: 1px solid var(--border-color);
            border-radius: 0.5rem;
            padding: 0.75rem 1rem;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            background-color: var(--input-bg);
            color: var(--text-light);
            width: 100%;
        }

        .form-control:focus {
            border-color: var(--primary-blue);
            box-shadow: 0 0 0 3px rgba(117, 175, 238, 0.2);
            outline: none;
        }

        .input-group {
            display: flex;
            align-items: stretch;
            width: 100%;
        }

        .input-group-text {
            display: flex;
            align-items: center;
            padding: 0.75rem 1rem;
            background-color: var(--medium-gray);
            border: 1px solid var(--border-color);
            border-radius: 0.5rem 0 0 0.5rem;
            color: var(--text-light);
            font-weight: 600;
        }

        .input-group .form-control {
            border-radius: 0 0.5rem 0.5rem 0;
            margin-left: -1px;
        }

        .form-text {
            font-size: 0.8rem;
            color: var(--text-dim);
            margin-top: 0.5rem;
        }

        .file-upload-container {
            position: relative;
            display: inline-block;
            width: 100%;
        }

        .file-upload-input {
            position: absolute;
            opacity: 0;
            width: 100%;
            height: 100%;
            cursor: pointer;
        }

        .file-upload-label {
            display: block;
            padding: 1.5rem;
            background-color: var(--medium-gray);
            border: 2px dashed var(--primary-blue);
            border-radius: 0.75rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .file-upload-label:hover {
            background-color: var(--light-gray);
        }

        .file-upload-label i {
            font-size: 2rem;
            color: var(--primary-blue);
            margin-bottom: 0.75rem;
            display: block;
        }

        .file-upload-label span {
            color: var(--text-light);
            font-weight: 500;
            display: block;
            margin-bottom: 0.5rem;
        }

        .file-upload-label small {
            color: var(--text-dim);
        }

        .deposit-btn {
            width: 100%;
            padding: 0.875rem;
            background: var(--primary-blue);
            color: var(--black);
            border: none;
            border-radius: 0.5rem;
            font-size: 1rem;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 1rem;
        }

        .deposit-btn:hover {
            background: var(--dark-blue);
            transform: translateY(-2px);
        }

        /* Deposit History Card */
        .deposit-history-card {
            background: var(--dark-gray);
            border-radius: 1rem;
            overflow: hidden;
            box-shadow: var(--card-shadow);
        }

        .deposit-history-title {
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--text-light);
            padding: 1.5rem;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            align-items: center;
        }

        .deposit-table {
            width: 100%;
            border-collapse: collapse;
        }

        .deposit-table th {
            background-color: var(--medium-gray);
            color: var(--text-light);
            font-weight: 600;
            padding: 1rem;
            text-align: left;
            font-size: 0.9rem;
        }

        .deposit-table td {
            padding: 1rem;
            border-bottom: 1px solid var(--border-color);
            font-size: 0.9rem;
        }

        .deposit-table tr:last-child td {
            border-bottom: none;
        }

        .deposit-table tr:hover {
            background-color: rgba(117, 175, 238, 0.05);
        }

        .status-badge {
            padding: 0.35rem 0.75rem;
            border-radius: 1rem;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .status-pending {
            background-color: var(--warning);
            color: var(--black);
        }

        .status-approved {
            background-color: var(--success);
            color: white;
        }

        .status-declined {
            background-color: var(--danger);
            color: white;
        }

        .view-proof-btn {
            background: var(--primary-blue);
            color: var(--black);
            border: none;
            border-radius: 0.5rem;
            padding: 0.5rem 0.75rem;
            font-size: 0.8rem;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .view-proof-btn:hover {
            background: var(--dark-blue);
        }

        .empty-state {
            text-align: center;
            padding: 3rem 1rem;
            color: var(--text-dim);
        }

        .empty-state i {
            font-size: 3rem;
            color: var(--light-gray);
            margin-bottom: 1rem;
        }

        .empty-state p {
            font-size: 1rem;
        }

        .info-alert {
            background: var(--dark-gray);
            border-radius: 1rem;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            box-shadow: var(--card-shadow);
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .info-alert i {
            font-size: 1.5rem;
            color: var(--primary-blue);
        }

        .info-alert p {
            margin: 0;
            color: var(--text-light);
        }

        /* Copy notification */
        .copy-notification {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background-color: var(--primary-blue);
            color: var(--black);
            padding: 12px 24px;
            border-radius: 8px;
            z-index: 9999;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            opacity: 1;
            transition: opacity 0.5s;
            font-weight: 600;
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .header {
                padding: 0.75rem 1rem;
            }
            
            .logo-text {
                font-size: 1.2rem;
            }
            
            .account-info {
                display: none;
            }
            
            .right-sidebar {
                width: 80%; /* 80% width on mobile instead of 50% */
                right: -80%;
            }
            
            .balance-amount {
                font-size: 2rem;
            }
            
            .amount-limits {
                flex-direction: column;
                gap: 0.75rem;
            }
            
            .deposit-table {
                font-size: 0.8rem;
            }
            
            .deposit-table th, .deposit-table td {
                padding: 0.75rem 0.5rem;
            }
            
            .bottom-nav-text {
                font-size: 0.6rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="logo">
            <img src="/g1.jpg" alt="Gainmart Logo">
            <div class="logo-text">Gainmart</div>
        </div>
        <div class="header-actions">
            <div class="account-section">
                <button class="account-btn" id="accountBtn">
                    <img src="/g1.jpg" alt="Profile" class="account-avatar">
                    <div class="account-info">
                        <span class="account-name"><?php echo $fullname ?></span>
                        <span class="account-username">@<?php echo $username ?></span>
                    </div>
                    <i class="fas fa-chevron-down"></i>
                </button>
                
                <div class="account-dropdown" id="accountDropdown">
                    <a href="settings" class="dropdown-item">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                    <div class="dropdown-divider"></div>
                    <a href="logout" class="dropdown-item">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Log Out</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Right Sidebar Menu -->
    <div class="right-sidebar" id="rightSidebar">
        <div class="sidebar-header">
            <img src="favicon.png" alt="Logo" class="sidebar-logo">
            <button class="sidebar-close" id="sidebarClose">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="sidebar-content">
            <div class="nk-sidebar-menu" data-simplebar>
                <ul class="nk-menu">
                    <li class="nk-menu-item">
                        <a href="../user/index" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-home"></i></span>
                            <span class="nk-menu-text">Dashboard</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_packages.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-box"></i></span>
                            <span class="nk-menu-text">Investment Packages</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_withdrawal.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-money-bill-wave"></i></span>
                            <span class="nk-menu-text">Withdraw Funds</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="referrals" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-users"></i></span>
                            <span class="nk-menu-text">My Referrals</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="settings" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-cog"></i></span>
                            <span class="nk-menu-text">Account Settings</span>
                        </a>
                    </li>
                    
                    <div class="mt-5"></div>
                </ul>
            </div>
        </div>
    </div>

    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <div class="nk-content px-0">
        <div class="container-fluid">
            <div class="nk-content-inner">
                <div class="nk-content-body">
                    <div class="row g-gs">

                        <div class="col-12 px-3 mt-4">
                            <h1 class="page-title">Deposit Funds</h1>
                        </div>

                        <?php if (!empty($successMsg)) echo $successMsg; ?>
                        <?php if (!empty($errorMsg)) echo $errorMsg; ?>
                        
                        <!-- Balance Card -->
                        <div class="col-12">
                            <div class="balance-card">
                                <div class="balance-label">Current Balance</div>
                                <div class="balance-amount">₦<?php echo number_format($currentFunds, 2); ?></div>
                            </div>
                        </div>
                        
                        <?php if ($bankData): ?>
                        <!-- Bank Details Card -->
                        <div class="col-12">
                            <div class="bank-details-card">
                                <div class="section-title">
                                    <i class="fas fa-university"></i>
                                    <span>Bank Details</span>
                                </div>
                                
                                <div class="bank-info-row">
                                    <div class="bank-info-label">Bank Name</div>
                                    <div class="bank-info-value"><?php echo htmlspecialchars($bankData['bank_name']); ?></div>
                                </div>
                                
                                <div class="bank-info-row">
                                    <div class="bank-info-label">Account Name</div>
                                    <div class="bank-info-value"><?php echo htmlspecialchars($bankData['account_name']); ?></div>
                                </div>
                                
                                <div class="bank-info-row">
                                    <div class="bank-info-label">Account Number</div>
                                    <div class="bank-info-value">
                                        <?php echo htmlspecialchars($bankData['account_number']); ?>
                                        <button class="copy-button" onclick="copyAccountNumber()">
                                            <i class="fas fa-copy"></i> Copy
                                        </button>
                                    </div>
                                </div>
                                
                                <div class="amount-limits">
                                    <div class="amount-limit">
                                        <div class="amount-limit-label">Minimum Amount</div>
                                        <div class="amount-limit-value">₦<?php echo number_format($bankData['min_amount'], 2); ?></div>
                                    </div>
                                    <div class="amount-limit">
                                        <div class="amount-limit-label">Maximum Amount</div>
                                        <div class="amount-limit-value">₦<?php echo number_format($bankData['max_amount'], 2); ?></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Deposit Form Card -->
                        <div class="col-12">
                            <div class="deposit-form-card">
                                <div class="section-title">
                                    <i class="fas fa-money-bill-wave"></i>
                                    <span>Submit Deposit</span>
                                </div>
                                
                                <form method="post" enctype="multipart/form-data">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label class="form-label">Amount (₦)</label>
                                                <div class="input-group">
                                                    <span class="input-group-text">₦</span>
                                                    <input type="number" class="form-control" name="amount" 
                                                           min="<?php echo $bankData['min_amount']; ?>" 
                                                           max="<?php echo $bankData['max_amount']; ?>" 
                                                           step="0.01" required>
                                                </div>
                                                <div class="form-text">
                                                    Between ₦<?php echo number_format($bankData['min_amount'], 2); ?> 
                                                    and ₦<?php echo number_format($bankData['max_amount'], 2); ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label class="form-label">Proof of Payment</label>
                                                <div class="file-upload-container">
                                                    <input type="file" class="file-upload-input" id="proof_image" name="proof_image" 
                                                           accept="image/*" required>
                                                    <label for="proof_image" class="file-upload-label">
                                                        <i class="fas fa-cloud-upload-alt"></i>
                                                        <span>Click to upload or drag and drop</span>
                                                        <small>Upload JPG, JPEG, PNG, or GIF file (Max 5MB)</small>
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <button type="submit" name="submit_deposit" class="deposit-btn">Submit Deposit</button>
                                </form>
                            </div>
                        </div>
                        
                        <!-- Deposit History Card -->
                        <div class="col-12">
                            <div class="deposit-history-card">
                                <div class="deposit-history-title">
                                    <i class="fas fa-history"></i>
                                    <span>Deposit History</span>
                                </div>
                                
                                <?php if ($depositsResult->num_rows > 0): ?>
                                <div class="table-responsive">
                                    <table class="deposit-table">
                                        <thead>
                                            <tr>
                                                <th>Transaction ID</th>
                                                <th>Amount</th>
                                                <th>Date</th>
                                                <th>Status</th>
                                                <th>Proof</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php while ($deposit = $depositsResult->fetch_assoc()): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($deposit['transaction_id']); ?></td>
                                                <td>₦<?php echo number_format($deposit['amount'], 2); ?></td>
                                                <td><?php echo date('M j, Y H:i', strtotime($deposit['created_at'])); ?></td>
                                                <td>
                                                    <?php if ($deposit['status'] == 'pending'): ?>
                                                    <span class="status-badge status-pending">Pending</span>
                                                    <?php elseif ($deposit['status'] == 'approved'): ?>
                                                    <span class="status-badge status-approved">Approved</span>
                                                    <?php else: ?>
                                                    <span class="status-badge status-declined">Declined</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if (!empty($deposit['proof_image'])): ?>
                                                    <a href="/stream/uploads/deposits/<?php echo $deposit['proof_image']; ?>" 
                                                       target="_blank" class="view-proof-btn">View Proof</a>
                                                    <?php else: ?>
                                                    <span class="text-muted">No proof</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endwhile; ?>
                                        </tbody>
                                    </table>
                                </div>
                                <?php else: ?>
                                <div class="empty-state">
                                    <i class="fas fa-inbox"></i>
                                    <p>No deposits found</p>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="col-12">
                            <div class="info-alert">
                                <i class="fas fa-info-circle"></i>
                                <p>Deposit is currently disabled. Please check back later.</p>
                            </div>
                        </div>
                        <?php endif; ?>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation Bar -->
    <div class="bottom-navigation">
        <a href="../user/index" class="bottom-nav-item">
            <i class="fas fa-home bottom-nav-icon"></i>
            <span class="bottom-nav-text">Dashboard</span>
        </a>
        <a href="user_packages.php" class="bottom-nav-item">
            <i class="fas fa-box bottom-nav-icon"></i>
            <span class="bottom-nav-text">Packages</span>
        </a>
        <a href="user_deposit.php" class="bottom-nav-item active">
            <i class="fas fa-money-bill-wave bottom-nav-icon"></i>
            <span class="bottom-nav-text">Deposit</span>
        </a>
        <a href="user_withdrawal.php" class="bottom-nav-item">
            <i class="fas fa-money-bill-wave bottom-nav-icon"></i>
            <span class="bottom-nav-text">Withdraw</span>
        </a>
        <a href="referrals" class="bottom-nav-item">
            <i class="fas fa-users bottom-nav-icon"></i>
            <span class="bottom-nav-text">Referrals</span>
        </a>
    </div>

    <script>
    // Account dropdown functionality
    document.addEventListener('DOMContentLoaded', function() {
        const accountBtn = document.getElementById('accountBtn');
        const accountDropdown = document.getElementById('accountDropdown');
        
        if (accountBtn && accountDropdown) {
            accountBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                accountDropdown.classList.toggle('active');
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', function() {
                accountDropdown.classList.remove('active');
            });
            
            // Prevent dropdown from closing when clicking inside it
            accountDropdown.addEventListener('click', function(e) {
                e.stopPropagation();
            });
        }
        
        // Right sidebar functionality
        const hamburgerBtn = document.getElementById('hamburgerBtn');
        const rightSidebar = document.getElementById('rightSidebar');
        const sidebarClose = document.getElementById('sidebarClose');
        const sidebarOverlay = document.getElementById('sidebarOverlay');
        
        if (hamburgerBtn && rightSidebar && sidebarClose && sidebarOverlay) {
            // Open sidebar when hamburger is clicked
            hamburgerBtn.addEventListener('click', function() {
                rightSidebar.classList.add('active');
                sidebarOverlay.classList.add('active');
                document.body.style.overflow = 'hidden'; // Prevent body scroll when sidebar is open
                
                // Reset animations when opening
                const menuItems = rightSidebar.querySelectorAll('.nk-menu-item');
                menuItems.forEach(item => {
                    item.style.animation = 'none';
                    item.offsetHeight; // Trigger reflow
                    item.style.animation = null;
                });
            });
            
            // Close sidebar when close button is clicked
            sidebarClose.addEventListener('click', function() {
                rightSidebar.classList.remove('active');
                sidebarOverlay.classList.remove('active');
                document.body.style.overflow = ''; // Restore body scroll
            });
            
            // Close sidebar when overlay is clicked
            sidebarOverlay.addEventListener('click', function() {
                rightSidebar.classList.remove('active');
                sidebarOverlay.classList.remove('active');
                document.body.style.overflow = ''; // Restore body scroll
            });
            
            // Handle submenu toggles
            const menuToggles = document.querySelectorAll('.nk-menu-toggle');
            menuToggles.forEach(toggle => {
                toggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    const parentItem = this.parentElement;
                    const submenu = parentItem.querySelector('.nk-menu-sub');
                    
                    if (submenu) {
                        submenu.classList.toggle('active');
                        this.classList.toggle('active');
                    }
                });
            });
        }
        
        // Set active navigation item based on current page
        const currentPath = window.location.pathname;
        const navItems = document.querySelectorAll('.bottom-nav-item');
        
        navItems.forEach(item => {
            const itemPath = item.getAttribute('href');
            if (currentPath.includes(itemPath)) {
                item.classList.add('active');
            } else {
                item.classList.remove('active');
            }
        });
    });

    function copyAccountNumber() {
        const accountNumber = '<?php echo htmlspecialchars($bankData['account_number']); ?>';
        navigator.clipboard.writeText(accountNumber).then(() => {
            // Create notification
            const notification = document.createElement('div');
            notification.className = 'copy-notification';
            notification.textContent = 'Account number copied to clipboard!';
            
            // Add the notification to the document
            document.body.appendChild(notification);
            
            // Remove notification after 3 seconds
            setTimeout(() => {
                notification.style.opacity = '0';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 500);
            }, 3000);
        }).catch(err => {
            console.error('Failed to copy: ', err);
        });
    }

    // File upload preview
    document.getElementById('proof_image').addEventListener('change', function() {
        const file = this.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                const label = document.querySelector('.file-upload-label');
                label.innerHTML = `
                    <img src="${e.target.result}" alt="Preview" style="max-width: 100%; max-height: 200px; border-radius: 8px; margin-bottom: 10px;">
                    <span>Click to change file</span>
                    <small>${file.name}</small>
                `;
            }
            reader.readAsDataURL(file);
        }
    });
    </script>
</body>
</html>