<?php

// Initialize variables
 $ptitle = "Withdraw Funds";
 $username = $_SESSION['username'] ?? '';
 $userId = null; // Will be set dynamically
 $successMsg = '';
 $errorMsg = '';
 $formSubmitted = false; // Track if form was submitted
 $showSuccessDialog = false; // Track if success dialog should be shown
 $transactionId = ''; // Store transaction ID for success dialog

// Include required files
require $_SERVER['DOCUMENT_ROOT']."/stream.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/includes/generalinclude.php";
require $_SERVER['DOCUMENT_ROOT']."$stream/user/includes/generalinclude.php";

// Set timezone
date_default_timezone_set('UTC');

// Get user ID from session or database
// First, try to get user ID from session
if (isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
} 
// If not in session, try to get from username in session
elseif (isset($_SESSION['username'])) {
    $username = $_SESSION['username'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE username = ?");
    $getUserQuery->bind_param("s", $username);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}
// If still not found, try to get from email in session
elseif (isset($_SESSION['email'])) {
    $email = $_SESSION['email'];
    // Get user ID from database
    $getUserQuery = $link->prepare("SELECT id FROM users WHERE email = ?");
    $getUserQuery->bind_param("s", $email);
    $getUserQuery->execute();
    $userResult = $getUserQuery->get_result();
    if ($userResult->num_rows > 0) {
        $userData = $userResult->fetch_assoc();
        $userId = $userData['id'];
        // Set user_id in session for future requests
        $_SESSION['user_id'] = $userId;
    }
}

// If user ID still not found, create a default user for testing
if (!$userId) {
    // Check if default user exists
    $checkUserQuery = $link->prepare("SELECT id FROM users WHERE username = 'Test User'");
    $checkUserQuery->execute();
    $checkResult = $checkUserQuery->get_result();
    
    if ($checkResult->num_rows == 0) {
        // Create default user
        $insertUserQuery = $link->prepare("INSERT INTO users (username, email, password, funds) VALUES (?, ?, ?, ?)");
        $defaultUsername = 'Test User';
        $defaultEmail = 'test@example.com';
        $defaultPassword = password_hash('password123', PASSWORD_DEFAULT);
        $defaultFunds = 0.00;
        $insertUserQuery->bind_param("sssd", $defaultUsername, $defaultEmail, $defaultPassword, $defaultFunds);
        $insertUserQuery->execute();
        $userId = $link->insert_id;
    } else {
        $userData = $checkResult->fetch_assoc();
        $userId = $userData['id'];
    }
    
    // Set user_id in session
    $_SESSION['user_id'] = $userId;
    $_SESSION['username'] = 'Test User';
    $_SESSION['email'] = 'test@example.com';
}

// Get withdrawal settings
 $settingsQuery = $link->query("SELECT setting_value FROM flow_settings WHERE setting_key = 'withdrawal_enabled'");
 $settingsResult = $settingsQuery->fetch_assoc();
 $withdrawalEnabled = $settingsResult['setting_value'] == 1 ? true : false;

// Get user funds from users table
 $userQuery = $link->prepare("SELECT funds FROM users WHERE id = ?");
 $userQuery->bind_param("i", $userId);
 $userQuery->execute();
 $userResult = $userQuery->get_result();

if ($userResult->num_rows > 0) {
    $user = $userResult->fetch_assoc();
    $currentFunds = $user['funds'];
} else {
    $currentFunds = 0;
}

// Get minimum withdrawal amount
 $minWithdrawalQuery = $link->query("SELECT setting_value FROM flow_settings WHERE setting_key = 'min_withdrawal'");
 $minWithdrawalResult = $minWithdrawalQuery->fetch_assoc();
 $minWithdrawal = $minWithdrawalResult['setting_value'] ?? 1000;

// Get maximum withdrawal amount setting
 $maxWithdrawalSettingQuery = $link->query("SELECT setting_value FROM flow_settings WHERE setting_key = 'max_withdrawal'");
 $maxWithdrawalSettingResult = $maxWithdrawalSettingQuery->fetch_assoc();
 $maxWithdrawalSetting = $maxWithdrawalSettingResult['setting_value'] ?? 1000000; // Default to 1,000,000

// Calculate effective maximum (lesser of user balance or admin-set maximum)
 $effectiveMax = min($currentFunds, $maxWithdrawalSetting);

// Define preset amounts
 $presetAmounts = [4000, 9000, 17000, 40000, 100000];

// Handle withdrawal submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_withdrawal'])) {
    $formSubmitted = true; // Mark form as submitted
    
    if (!$withdrawalEnabled) {
        $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Withdrawals are currently disabled. Please try again later.</div>';
    } else {
        $amount = $_POST['amount'] ?? 0;
        $method = $_POST['method'] ?? '';
        $details = $_POST['details'] ?? '';
        
        // Validate amount
        if ($amount < $minWithdrawal) {
            $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Minimum withdrawal amount is ₦' . number_format($minWithdrawal, 2) . '</div>';
        } elseif ($amount > $effectiveMax) {
            $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Maximum withdrawal amount is ₦' . number_format($effectiveMax, 2) . '</div>';
        } elseif (empty($method) || empty($details)) {
            $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Please fill all required fields.</div>';
        } else {
            // Generate unique transaction ID
            $transactionId = 'WD' . time() . rand(1000, 9999);
            
            // Insert withdrawal request
            $insertWithdrawal = $link->prepare("INSERT INTO flow_withdrawal_requests (user_id, transaction_id, amount, method, details, status) VALUES (?, ?, ?, ?, ?, 'pending')");
            $insertWithdrawal->bind_param("issss", $userId, $transactionId, $amount, $method, $details);
            
            if ($insertWithdrawal->execute()) {
                // Set variables for success dialog
                $showSuccessDialog = true;
                $_SESSION['showSuccessDialog'] = true;
                $_SESSION['transactionId'] = $transactionId;
                $_SESSION['withdrawalAmount'] = $amount;
                
                // Clear form data by redirecting to the same page
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            } else {
                $errorMsg = '<div class="copy-notification" style="background-color: var(--danger);">Error submitting withdrawal request. Please try again.</div>';
            }
        }
    }
}

// Check if we should show success dialog from session
if (isset($_SESSION['showSuccessDialog']) && $_SESSION['showSuccessDialog']) {
    $showSuccessDialog = true;
    $transactionId = $_SESSION['transactionId'] ?? '';
    $withdrawalAmount = $_SESSION['withdrawalAmount'] ?? 0;
    
    // Clear session variables
    unset($_SESSION['showSuccessDialog']);
    unset($_SESSION['transactionId']);
    unset($_SESSION['withdrawalAmount']);
}

// Get user's withdrawal history
 $withdrawalsQuery = $link->prepare("SELECT * FROM flow_withdrawal_requests WHERE user_id = ? ORDER BY created_at DESC");
 $withdrawalsQuery->bind_param("i", $userId);
 $withdrawalsQuery->execute();
 $withdrawalsResult = $withdrawalsQuery->get_result();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <meta name="description" content="<?php echo $sitedesc ?>">
    <title><?php echo $ptitle ?> - <?php echo $sitename ?></title>
    <link rel="shortcut icon" href="../favicon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css"/>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        :root {
            /* Gainmart colors */
            --primary-blue: #75AFEE;
            --dark-blue: #4A8FE7;
            --darker-blue: #2E6CBF;
            --black: #000000;
            --dark-gray: #1A1A1A;
            --medium-gray: #2A2A2A;
            --light-gray: #3A3A3A;
            --text-light: #E0E0E0;
            --text-dim: #A0A0A0;
            --card-bg: #1F1F1F;
            --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            --input-bg: #2A2A2A;
            --border-color: #3A3A3A;
            --success: #4CAF50;
            --warning: #FFC107;
            --danger: #F44336;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--black);
            color: var(--text-light);
            transition: all 0.3s ease;
            min-height: 100vh;
            padding-bottom: 5rem; /* Added padding for bottom nav */
        }

        h1,h2,h3,h4,h5,h6 {
            font-weight: 700;
        }
        
        .container-fluid {
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
            margin-right: auto;
            margin-left: auto;
        }

        .row {
            display: flex;
            flex-wrap: wrap;
            margin-right: -15px;
            margin-left: -15px;
        }

        .col-12 {
            position: relative;
            width: 100%;
            padding-right: 15px;
            padding-left: 15px;
        }

        .g-gs {
            --bs-gutter-x: 1.5rem;
            --bs-gutter-y: 0;
            display: flex;
            flex-wrap: wrap;
            margin-top: calc(-1 * var(--bs-gutter-y));
            margin-right: calc(-.5 * var(--bs-gutter-x));
            margin-left: calc(-.5 * var(--bs-gutter-x));
        }

        .g-gs > * {
            flex-shrink: 0;
            width: 100%;
            max-width: 100%;
            padding-right: calc(var(--bs-gutter-x) * .5);
            padding-left: calc(var(--bs-gutter-x) * .5);
            margin-top: var(--bs-gutter-y);
        }

        .mt-3 {
            margin-top: 1rem !important;
        }

        .mt-4 {
            margin-top: 1.5rem !important;
        }

        .px-3 {
            padding-left: 1rem !important;
            padding-right: 1rem !important;
        }

        .mb-3 {
            margin-bottom: 1rem !important;
        }

        .mr-2 {
            margin-right: 0.5rem !important;
        }

        /* Header Styles */
        .header {
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            padding: 1rem 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .logo img {
            height: 40px;
        }

        .logo-text {
            font-size: 1.5rem;
            font-weight: 800;
            color: var(--primary-blue);
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .notification-btn {
            position: relative;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.2rem;
            cursor: pointer;
            padding: 0.75rem;
            border-radius: 50%;
            transition: all 0.3s ease;
        }

        .notification-btn:hover {
            background: var(--light-gray);
            color: var(--primary-blue);
        }

        .notification-badge {
            position: absolute;
            top: 0;
            right: 0;
            width: 10px;
            height: 10px;
            background: var(--danger);
            border-radius: 50%;
        }

        .account-section {
            position: relative;
        }

        .account-btn {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            cursor: pointer;
            padding: 0.5rem 1rem;
            border-radius: 2rem;
            transition: all 0.3s ease;
        }

        .account-btn:hover {
            background: var(--light-gray);
        }

        .account-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--primary-blue);
        }

        .account-info {
            display: flex;
            flex-direction: column;
            align-items: flex-start;
        }

        .account-name {
            font-weight: 600;
            font-size: 0.9rem;
        }

        .account-username {
            font-size: 0.8rem;
            color: var(--text-dim);
        }

        .account-dropdown {
            position: absolute;
            top: 100%;
            right: 0;
            margin-top: 0.5rem;
            background: var(--dark-gray);
            backdrop-filter: blur(10px);
            border-radius: 0.75rem;
            box-shadow: 0 10px 25px rgba(0,0,0,0.4);
            min-width: 220px;
            overflow: hidden;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.3s ease;
        }

        .account-dropdown.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }

        .dropdown-divider {
            height: 1px;
            background: var(--light-gray);
            margin: 0.5rem 0;
        }

        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: var(--text-light);
            text-decoration: none;
            transition: background-color 0.3s;
        }

        .dropdown-item:hover {
            background-color: var(--medium-gray);
        }

        .dropdown-item i {
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: var(--primary-blue);
        }

        /* Bottom Navigation Bar */
        .bottom-navigation {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: var(--dark-gray);
            box-shadow: 0 -2px 10px rgba(0,0,0,0.2);
            z-index: 100;
            padding: 0.75rem 0;
            display: flex;
            justify-content: space-around;
            align-items: center;
        }

        .bottom-nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
            color: var(--text-dim);
            text-decoration: none;
            width: 20%;
        }

        .bottom-nav-item:hover {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-item.active {
            background: var(--medium-gray);
            color: var(--primary-blue);
        }

        .bottom-nav-icon {
            font-size: 1.5rem;
            margin-bottom: 0.25rem;
        }

        .bottom-nav-text {
            font-size: 0.7rem;
            font-weight: 600;
        }

        /* Right Sidebar Menu */
        .right-sidebar {
            position: fixed;
            top: 0;
            right: -50%; /* Hidden by default - half page width */
            width: 50%; /* Half page width */
            height: 100%;
            background: var(--dark-gray);
            z-index: 1001;
            transition: right 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
            overflow-y: auto;
            box-shadow: -5px 0 25px rgba(0,0,0,0.4);
        }

        .right-sidebar.active {
            right: 0;
        }

        .sidebar-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem;
            border-bottom: 1px solid var(--light-gray);
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
        }

        .sidebar-logo {
            height: 40px;
            filter: drop-shadow(0 2px 4px rgba(0,0,0,0.3));
        }

        .sidebar-close {
            background: var(--medium-gray);
            border: none;
            color: var(--text-light);
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
        }

        .sidebar-close:hover {
            background: var(--light-gray);
            transform: rotate(90deg);
        }

        .sidebar-content {
            padding: 1rem 0;
            position: relative;
            z-index: 2;
            background: var(--dark-gray);
            min-height: calc(100% - 73px); /* Subtract header height */
        }

        .nk-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .nk-menu-item {
            margin: 0.25rem 0;
            opacity: 0;
            transform: translateX(20px);
            animation: slideInRight 0.5s forwards;
        }

        /* Stagger animation for menu items */
        .nk-menu-item:nth-child(1) { animation-delay: 0.1s; }
        .nk-menu-item:nth-child(2) { animation-delay: 0.15s; }
        .nk-menu-item:nth-child(3) { animation-delay: 0.2s; }
        .nk-menu-item:nth-child(4) { animation-delay: 0.25s; }
        .nk-menu-item:nth-child(5) { animation-delay: 0.3s; }
        .nk-menu-item:nth-child(6) { animation-delay: 0.35s; }
        .nk-menu-item:nth-child(7) { animation-delay: 0.4s; }
        .nk-menu-item:nth-child(8) { animation-delay: 0.45s; }
        .nk-menu-item:nth-child(9) { animation-delay: 0.5s; }
        .nk-menu-item:nth-child(10) { animation-delay: 0.55s; }
        .nk-menu-item:nth-child(11) { animation-delay: 0.6s; }
        .nk-menu-item:nth-child(12) { animation-delay: 0.65s; }
        .nk-menu-item:nth-child(13) { animation-delay: 0.7s; }
        .nk-menu-item:nth-child(14) { animation-delay: 0.75s; }
        .nk-menu-item:nth-child(15) { animation-delay: 0.8s; }

        @keyframes slideInRight {
            from {
                opacity: 0;
                transform: translateX(20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .nk-menu-link {
            display: flex;
            align-items: center;
            padding: 0.9rem 1.5rem;
            color: var(--text-light);
            text-decoration: none;
            transition: all 0.3s ease;
            border-radius: 0.75rem;
            margin: 0 0.75rem;
            position: relative;
            overflow: hidden;
        }

        .nk-menu-link::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary-blue);
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover {
            background: var(--medium-gray);
            transform: translateX(5px);
        }

        .nk-menu-link:hover::before {
            transform: scaleY(1);
        }

        .nk-menu-icon {
            margin-right: 1rem;
            color: var(--primary-blue);
            font-size: 1.2rem;
            width: 24px;
            text-align: center;
            transition: transform 0.3s ease;
        }

        .nk-menu-link:hover .nk-menu-icon {
            transform: scale(1.1);
        }

        .nk-menu-text {
            font-weight: 500;
            font-size: 1rem;
        }

        .nk-menu-heading {
            padding: 0.75rem 1.5rem;
            margin: 1.5rem 0 0.75rem;
            color: var(--text-dim);
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.1em;
            position: relative;
        }

        .nk-menu-heading::after {
            content: '';
            position: absolute;
            bottom: -0.5rem;
            left: 1.5rem;
            right: 1.5rem;
            height: 1px;
            background: var(--light-gray);
        }

        .overline-title {
            font-weight: 600;
        }

        .nk-menu-sub {
            list-style: none;
            padding: 0;
            margin: 0.25rem 0 0.25rem 2rem;
            background: var(--black);
            border-radius: 0.75rem;
            overflow: hidden;
            max-height: 0;
            opacity: 0;
            transition: all 0.4s ease;
        }

        .nk-menu-sub.active {
            max-height: 500px;
            opacity: 1;
        }

        .nk-menu-sub .nk-menu-item {
            margin: 0;
            opacity: 1;
            transform: none;
            animation: none;
        }

        .nk-menu-sub .nk-menu-link {
            padding: 0.7rem 1rem 0.7rem 1.5rem;
            font-size: 0.9rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
        }

        .nk-menu-sub .nk-menu-link::before {
            width: 3px;
        }

        .nk-menu-sub .nk-menu-icon {
            font-size: 1rem;
            margin-right: 0.75rem;
        }

        /* Menu toggle icon animation */
        .nk-menu-toggle .nk-menu-icon::after {
            content: '\f107';
            font-family: 'Font Awesome 6 Free';
            font-weight: 900;
            margin-left: auto;
            transition: transform 0.3s ease;
        }

        .nk-menu-toggle.active .nk-menu-icon::after {
            transform: rotate(180deg);
        }

        /* Overlay for sidebar */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        /* Page Title */
        .page-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.5rem;
            padding: 0 1rem;
        }

        /* Withdrawal Container */
        .withdrawal-container {
            padding: 0 1rem;
        }

        /* Balance Card */
        .balance-card {
            background: linear-gradient(135deg, var(--dark-gray) 0%, var(--medium-gray) 100%);
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
            margin-bottom: 1.5rem;
            position: relative;
            overflow: hidden;
        }

        .balance-card::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 150px;
            height: 150px;
            background: var(--primary-blue);
            opacity: 0.1;
            border-radius: 50%;
            transform: translate(50px, -50px);
        }

        .balance-card::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100px;
            height: 100px;
            background: var(--primary-blue);
            opacity: 0.1;
            border-radius: 50%;
            transform: translate(-30px, 30px);
        }

        .balance-label {
            font-size: 0.9rem;
            font-weight: 600;
            color: var(--text-dim);
            margin-bottom: 0.5rem;
            position: relative;
            z-index: 1;
        }

        .balance-amount {
            font-size: 2.5rem;
            font-weight: 800;
            color: var(--primary-blue);
            margin-bottom: 1rem;
            position: relative;
            z-index: 1;
        }

        .balance-status {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 2rem;
            position: relative;
            z-index: 1;
        }

        .status-dot {
            width: 10px;
            height: 10px;
            border-radius: 50%;
        }

        .status-dot.enabled {
            background: var(--success);
        }

        .status-dot.disabled {
            background: var(--danger);
        }

        .status-text {
            font-size: 0.85rem;
            font-weight: 600;
            color: var(--text-light);
        }

        /* Withdrawal Form */
        .withdrawal-form {
            background: var(--dark-gray);
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
            margin-bottom: 1.5rem;
        }

        .section-title {
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .section-title i {
            color: var(--primary-blue);
        }

        .form-group {
            margin-bottom: 1.25rem;
        }

        .form-label {
            display: block;
            font-size: 0.9rem;
            font-weight: 600;
            color: var(--text-light);
            margin-bottom: 0.5rem;
        }

        .form-control {
            width: 100%;
            padding: 0.875rem 1rem;
            background: var(--input-bg);
            border: 1px solid var(--border-color);
            border-radius: 0.5rem;
            color: var(--text-light);
            font-size: 0.9rem;
            transition: all 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary-blue);
            box-shadow: 0 0 0 3px rgba(117, 175, 238, 0.2);
        }

        .form-control:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .form-control[readonly] {
            background-color: var(--light-gray);
            opacity: 0.8;
        }

        textarea.form-control {
            resize: vertical;
            min-height: 100px;
        }

        .form-hint {
            font-size: 0.8rem;
            color: var(--text-dim);
            margin-top: 0.5rem;
        }

        .amount-limits {
            background: var(--medium-gray);
            border-radius: 0.5rem;
            padding: 1rem;
            margin-top: 0.75rem;
        }

        .amount-limits-title {
            font-size: 0.85rem;
            font-weight: 600;
            color: var(--primary-blue);
            margin-bottom: 0.75rem;
        }

        .amount-limits-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }

        .amount-limit {
            text-align: center;
        }

        .amount-limit-label {
            font-size: 0.75rem;
            color: var(--text-dim);
            margin-bottom: 0.25rem;
        }

        .amount-limit-value {
            font-size: 1rem;
            font-weight: 700;
            color: var(--text-light);
        }

        /* Preset Amount Buttons */
        .preset-amounts {
            display: flex;
            flex-wrap: wrap;
            gap: 0.75rem;
            margin-bottom: 1rem;
        }

        .preset-amount-btn {
            flex: 1;
            min-width: calc(33.333% - 0.5rem);
            padding: 0.75rem;
            background: var(--medium-gray);
            border: 1px solid var(--border-color);
            border-radius: 0.5rem;
            color: var(--text-light);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
            font-size: 1rem;
        }

        .preset-amount-btn:hover {
            background: var(--primary-blue);
            color: var(--black);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .preset-amount-btn.active {
            background: var(--primary-blue);
            color: var(--black);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .withdrawal-btn {
            width: 100%;
            padding: 0.875rem;
            background: var(--primary-blue);
            color: var(--black);
            border: none;
            border-radius: 0.5rem;
            font-size: 1rem;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .withdrawal-btn:hover {
            background: var(--dark-blue);
            transform: translateY(-2px);
        }

        .withdrawal-btn:disabled {
            background: var(--light-gray);
            color: var(--text-dim);
            cursor: not-allowed;
            transform: none;
        }

        .disabled-message {
            background: var(--danger);
            color: white;
            padding: 1rem;
            border-radius: 0.5rem;
            margin-bottom: 1rem;
            text-align: center;
            font-weight: 600;
        }

        /* Withdrawal Rules */
        .withdrawal-rules {
            background: var(--dark-gray);
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
            margin-bottom: 1.5rem;
        }

        .rules-list {
            list-style: none;
        }

        .rules-list li {
            position: relative;
            padding-left: 1.75rem;
            margin-bottom: 0.75rem;
            font-size: 0.9rem;
            color: var(--text-light);
        }

        .rules-list li:last-child {
            margin-bottom: 0;
        }

        .rules-list li::before {
            content: '\f00c';
            font-family: 'Font Awesome 6 Free';
            font-weight: 900;
            position: absolute;
            left: 0;
            color: var(--primary-blue);
        }

        /* Withdrawal History */
        .withdrawal-history {
            background: var(--dark-gray);
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
        }

        .history-tabs {
            display: flex;
            margin-bottom: 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }

        .history-tab {
            padding: 0.75rem 1rem;
            font-size: 0.9rem;
            font-weight: 600;
            color: var(--text-dim);
            border-bottom: 2px solid transparent;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .history-tab.active {
            color: var(--primary-blue);
            border-bottom-color: var(--primary-blue);
        }

        .history-content {
            display: none;
        }

        .history-content.active {
            display: block;
        }

        .history-item {
            background: var(--medium-gray);
            border-radius: 0.75rem;
            padding: 1rem;
            margin-bottom: 1rem;
            transition: all 0.3s ease;
        }

        .history-item:last-child {
            margin-bottom: 0;
        }

        .history-item:hover {
            transform: translateY(-3px);
        }

        .history-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }

        .history-amount {
            font-size: 1.1rem;
            font-weight: 700;
            color: var(--primary-blue);
        }

        .history-status {
            padding: 0.25rem 0.75rem;
            border-radius: 1rem;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .history-status.pending {
            background: rgba(255, 193, 7, 0.2);
            color: var(--warning);
        }

        .history-status.approved {
            background: rgba(76, 175, 80, 0.2);
            color: var(--success);
        }

        .history-status.declined {
            background: rgba(244, 67, 54, 0.2);
            color: var(--danger);
        }

        .history-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 0.75rem;
            margin-bottom: 0.75rem;
        }

        .history-detail {
            font-size: 0.85rem;
        }

        .history-detail-label {
            color: var(--text-dim);
            margin-bottom: 0.25rem;
        }

        .history-detail-value {
            color: var(--text-light);
            font-weight: 500;
        }

        .history-account {
            background: var(--light-gray);
            border-radius: 0.5rem;
            padding: 0.75rem;
            font-size: 0.85rem;
            color: var(--text-light);
        }

        .history-account-label {
            color: var(--text-dim);
            margin-bottom: 0.25rem;
            font-weight: 600;
        }

        .empty-state {
            text-align: center;
            padding: 2rem 1rem;
            color: var(--text-dim);
        }

        .empty-state-icon {
            font-size: 2.5rem;
            color: var(--light-gray);
            margin-bottom: 1rem;
        }

        .empty-state-text {
            font-size: 1rem;
        }

        /* Copy notification */
        .copy-notification {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background-color: var(--primary-blue);
            color: var(--black);
            padding: 12px 24px;
            border-radius: 8px;
            z-index: 9999;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            opacity: 1;
            transition: opacity 0.5s;
            font-weight: 600;
        }

        /* Success Dialog */
        .success-dialog-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 10000;
            display: flex;
            justify-content: center;
            align-items: center;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .success-dialog-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        .success-dialog {
            background: var(--dark-gray);
            border-radius: 1rem;
            width: 90%;
            max-width: 500px;
            padding: 2rem;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.5);
            transform: scale(0.8);
            transition: transform 0.3s ease;
            text-align: center;
        }

        .success-dialog-overlay.active .success-dialog {
            transform: scale(1);
        }

        .success-icon {
            width: 80px;
            height: 80px;
            background: var(--success);
            border-radius: 50%;
            display: flex;
            justify-content: center;
            align-items: center;
            margin: 0 auto 1.5rem;
        }

        .success-icon i {
            font-size: 2.5rem;
            color: var(--black);
        }

        .success-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-light);
            margin-bottom: 1rem;
        }

        .success-message {
            font-size: 1rem;
            color: var(--text-dim);
            margin-bottom: 1.5rem;
            line-height: 1.5;
        }

        .success-details {
            background: var(--medium-gray);
            border-radius: 0.75rem;
            padding: 1rem;
            margin-bottom: 1.5rem;
            text-align: left;
        }

        .success-detail-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.75rem;
        }

        .success-detail-row:last-child {
            margin-bottom: 0;
        }

        .success-detail-label {
            color: var(--text-dim);
            font-size: 0.9rem;
        }

        .success-detail-value {
            color: var(--text-light);
            font-weight: 600;
            font-size: 0.9rem;
        }

        .success-close-btn {
            background: var(--primary-blue);
            color: var(--black);
            border: none;
            border-radius: 0.5rem;
            padding: 0.75rem 2rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .success-close-btn:hover {
            background: var(--dark-blue);
            transform: translateY(-2px);
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .header {
                padding: 0.75rem 1rem;
            }
            
            .logo-text {
                font-size: 1.2rem;
            }
            
            .account-info {
                display: none;
            }
            
            .right-sidebar {
                width: 80%; /* 80% width on mobile instead of 50% */
                right: -80%;
            }
            
            .balance-amount {
                font-size: 2rem;
            }
            
            .amount-limits-grid {
                grid-template-columns: 1fr;
                gap: 0.75rem;
            }
            
            .history-details {
                grid-template-columns: 1fr;
            }
            
            .bottom-nav-text {
                font-size: 0.6rem;
            }
            
            .preset-amount-btn {
                min-width: calc(50% - 0.375rem);
            }
            
            .success-dialog {
                padding: 1.5rem;
            }
            
            .success-icon {
                width: 60px;
                height: 60px;
                margin-bottom: 1rem;
            }
            
            .success-icon i {
                font-size: 2rem;
            }
            
            .success-title {
                font-size: 1.25rem;
                margin-bottom: 0.75rem;
            }
            
            .success-message {
                font-size: 0.9rem;
                margin-bottom: 1rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="logo">
            <img src="/g1.jpg" alt="Gainmart Logo">
            <div class="logo-text">Gainmart</div>
        </div>
        <div class="header-actions">
            <div class="account-section">
                <button class="account-btn" id="accountBtn">
                    <img src="/g1.jpg" alt="Profile" class="account-avatar">
                    <div class="account-info">
                        <span class="account-name"><?php echo $fullname ?></span>
                        <span class="account-username">@<?php echo $username ?></span>
                    </div>
                    <i class="fas fa-chevron-down"></i>
                </button>
                
                <div class="account-dropdown" id="accountDropdown">
                    <a href="settings" class="dropdown-item">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                    <div class="dropdown-divider"></div>
                    <a href="logout" class="dropdown-item">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Log Out</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Right Sidebar Menu -->
    <div class="right-sidebar" id="rightSidebar">
        <div class="sidebar-header">
            <img src="favicon.png" alt="Logo" class="sidebar-logo">
            <button class="sidebar-close" id="sidebarClose">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="sidebar-content">
            <div class="nk-sidebar-menu" data-simplebar>
                <ul class="nk-menu">
                    <li class="nk-menu-item">
                        <a href="../user/index" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-home"></i></span>
                            <span class="nk-menu-text">Dashboard</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_packages.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-box"></i></span>
                            <span class="nk-menu-text">Investment Packages</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="user_withdrawal.php" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-money-bill-wave"></i></span>
                            <span class="nk-menu-text">Withdraw Funds</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="referrals" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-users"></i></span>
                            <span class="nk-menu-text">My Referrals</span>
                        </a>
                    </li>
                    <li class="nk-menu-item">
                        <a href="settings" class="nk-menu-link">
                            <span class="nk-menu-icon"><i class="fas fa-cog"></i></span>
                            <span class="nk-menu-text">Account Settings</span>
                        </a>
                    </li>
                    
                    <div class="mt-5"></div>
                </ul>
            </div>
        </div>
    </div>

    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Success Dialog -->
    <div class="success-dialog-overlay" id="successDialogOverlay">
        <div class="success-dialog">
            <div class="success-icon">
                <i class="fas fa-check"></i>
            </div>
            <h3 class="success-title">Withdrawal Successful!</h3>
            <p class="success-message">Your withdrawal request has been successfully placed and is now being processed.</p>
            
            <div class="success-details">
                <div class="success-detail-row">
                    <span class="success-detail-label">Transaction ID:</span>
                    <span class="success-detail-value"><?php echo $transactionId; ?></span>
                </div>
                <div class="success-detail-row">
                    <span class="success-detail-label">Amount:</span>
                    <span class="success-detail-value">₦<?php echo number_format($withdrawalAmount, 2); ?></span>
                </div>
                <div class="success-detail-row">
                    <span class="success-detail-label">Status:</span>
                    <span class="success-detail-value">Pending</span>
                </div>
                <div class="success-detail-row">
                    <span class="success-detail-label">Processing Time:</span>
                    <span class="success-detail-value">24-48 hours</span>
                </div>
            </div>
            
            <button class="success-close-btn" id="successCloseBtn">OK</button>
        </div>
    </div>

    <div class="nk-content px-0">
        <div class="container-fluid">
            <div class="nk-content-inner">
                <div class="nk-content-body">
                    <div class="row g-gs">

                        <div class="col-12 px-3 mt-4">
                            <h1 class="page-title">Withdraw Funds</h1>
                        </div>

                        <?php if (isset($successMsg)) echo $successMsg; ?>
                        <?php if (isset($errorMsg)) echo $errorMsg; ?>
                        
                        <!-- Balance Card -->
                        <div class="col-12">
                            <div class="balance-card">
                                <div class="balance-label">Available Balance</div>
                                <div class="balance-amount">₦<?php echo number_format($currentFunds, 2); ?></div>
                                <div class="balance-status">
                                    <div class="status-dot <?php echo $withdrawalEnabled ? 'enabled' : 'disabled'; ?>"></div>
                                    <div class="status-text">Withdrawals <?php echo $withdrawalEnabled ? 'Enabled' : 'Disabled'; ?></div>
                                </div>
                            </div>
                        </div>

                        <!-- Withdrawal Form -->
                        <div class="col-12">
                            <div class="withdrawal-form">
                                <h3 class="section-title">
                                    <i class="fas fa-money-bill-wave"></i>
                                    Request Withdrawal
                                </h3>
                                
                                <?php if (!$withdrawalEnabled): ?>
                                <div class="disabled-message">
                                    Withdrawals are currently disabled. Please check back later.
                                </div>
                                <?php endif; ?>
                                
                                <form method="POST" id="withdrawalForm">
                                    <div class="form-group">
                                        <label class="form-label">Select Amount (₦)</label>
                                        
                                        <!-- Preset Amount Buttons -->
                                        <div class="preset-amounts">
                                            <?php foreach ($presetAmounts as $amount): ?>
                                                <?php if ($amount >= $minWithdrawal && $amount <= $effectiveMax): ?>
                                                    <button type="button" class="preset-amount-btn" data-amount="<?php echo $amount; ?>" <?php echo !$withdrawalEnabled ? 'disabled' : ''; ?>>
                                                        ₦<?php echo number_format($amount, 0); ?>
                                                    </button>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </div>
                                        
                                        <input type="number" class="form-control" name="amount" id="amountInput" min="<?php echo $minWithdrawal ?>" max="<?php echo $effectiveMax ?>" step="0.01" required readonly <?php echo !$withdrawalEnabled ? 'disabled' : ''; ?>>
                                        
                                        <div class="amount-limits">
                                            <div class="amount-limits-title">Withdrawal Limits</div>
                                            <div class="amount-limits-grid">
                                                <div class="amount-limit">
                                                    <div class="amount-limit-label">Minimum</div>
                                                    <div class="amount-limit-value">₦<?php echo number_format($minWithdrawal, 2) ?></div>
                                                </div>
                                                <div class="amount-limit">
                                                    <div class="amount-limit-label">Maximum</div>
                                                    <div class="amount-limit-value">₦<?php echo number_format($maxWithdrawalSetting, 2) ?></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label">Withdrawal Method</label>
                                        <input type="hidden" name="method" value="Bank Transfer" <?php echo !$withdrawalEnabled ? 'disabled' : ''; ?>>
                                        <div class="form-control" style="background-color: var(--input-bg); cursor: default;">Bank Transfer</div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label">Bank Account Details</label>
                                        <textarea class="form-control" name="details" rows="3" placeholder="Enter your bank account details (account number, bank name, account name)" required <?php echo !$withdrawalEnabled ? 'disabled' : ''; ?>></textarea>
                                        <div class="form-hint">Please provide accurate bank details to avoid delays</div>
                                    </div>
                                    
                                    <button type="submit" name="submit_withdrawal" class="withdrawal-btn" <?php echo !$withdrawalEnabled ? 'disabled' : ''; ?>>
                                        Submit Withdrawal Request
                                    </button>
                                </form>
                            </div>
                        </div>

                        <!-- Withdrawal Rules -->
                        <div class="col-12">
                            <div class="withdrawal-rules">
                                <h3 class="section-title">
                                    <i class="fas fa-list-check"></i>
                                    Withdrawal Rules
                                </h3>
                                
                                <ul class="rules-list">
                                    <li>Minimum withdrawal amount: ₦<?php echo number_format($minWithdrawal, 2) ?></li>
                                    <li>Maximum withdrawal amount: ₦<?php echo number_format($maxWithdrawalSetting, 2) ?></li>
                                    <li>Processing time: 24-48 hours (excluding weekends and holidays)</li>
                                    <li>Withdrawals can only be made to verified bank accounts</li>
                                    <li>Approved withdrawals are non-reversible</li>
                                    <li>Withdrawal requests may be subject to review</li>
                                    <li>Ensure you provide accurate bank account details</li>
                                </ul>
                            </div>
                        </div>

                        <!-- Withdrawal History -->
                        <div class="col-12">
                            <div class="withdrawal-history">
                                <h3 class="section-title">
                                    <i class="fas fa-history"></i>
                                    Withdrawal History
                                </h3>
                                
                                <?php if ($withdrawalsResult->num_rows > 0): ?>
                                    <div class="history-list">
                                        <?php while ($withdrawal = $withdrawalsResult->fetch_assoc()): ?>
                                            <?php
                                                $id = $withdrawal['id'];
                                                $transactionId = $withdrawal['transaction_id'];
                                                $amount = $withdrawal['amount'];
                                                $method = $withdrawal['method'];
                                                $details = $withdrawal['details'];
                                                $status = $withdrawal['status'];
                                                $created_at = date('M j, Y H:i', strtotime($withdrawal['created_at']));
                                                $adminNotes = $withdrawal['admin_notes'];
                                                
                                                $statusClass = '';
                                                if ($status == 'pending') {
                                                    $statusClass = 'pending';
                                                } elseif ($status == 'approved') {
                                                    $statusClass = 'approved';
                                                } else {
                                                    $statusClass = 'declined';
                                                }
                                            ?>
                                            <div class="history-item">
                                                <div class="history-header">
                                                    <div class="history-amount">₦<?php echo number_format($amount, 2) ?></div>
                                                    <div class="history-status <?php echo $statusClass ?>"><?php echo ucfirst($status) ?></div>
                                                </div>
                                                
                                                <div class="history-details">
                                                    <div class="history-detail">
                                                        <div class="history-detail-label">Transaction ID</div>
                                                        <div class="history-detail-value"><?php echo $transactionId ?></div>
                                                    </div>
                                                    <div class="history-detail">
                                                        <div class="history-detail-label">Date</div>
                                                        <div class="history-detail-value"><?php echo $created_at ?></div>
                                                    </div>
                                                    <div class="history-detail">
                                                        <div class="history-detail-label">Method</div>
                                                        <div class="history-detail-value"><?php echo $method ?></div>
                                                    </div>
                                                    <?php if (!empty($adminNotes)): ?>
                                                    <div class="history-detail">
                                                        <div class="history-detail-label">Admin Notes</div>
                                                        <div class="history-detail-value"><?php echo htmlspecialchars($adminNotes) ?></div>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <div class="history-account">
                                                    <div class="history-account-label">Bank Account Details:</div>
                                                    <?php echo nl2br(htmlspecialchars($details)) ?>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                <?php else: ?>
                                    <div class="empty-state">
                                        <div class="empty-state-icon">
                                            <i class="fas fa-receipt"></i>
                                        </div>
                                        <div class="empty-state-text">No withdrawal history found</div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation Bar -->
    <div class="bottom-navigation">
        <a href="../user/index" class="bottom-nav-item">
            <i class="fas fa-home bottom-nav-icon"></i>
            <span class="bottom-nav-text">Dashboard</span>
        </a>
        <a href="user_packages.php" class="bottom-nav-item">
            <i class="fas fa-box bottom-nav-icon"></i>
            <span class="bottom-nav-text">Packages</span>
        </a>
        <a href="user_withdrawal.php" class="bottom-nav-item active">
            <i class="fas fa-money-bill-wave bottom-nav-icon"></i>
            <span class="bottom-nav-text">Withdraw</span>
        </a>
        <a href="referrals" class="bottom-nav-item">
            <i class="fas fa-users bottom-nav-icon"></i>
            <span class="bottom-nav-text">Referrals</span>
        </a>
        <a href="settings" class="bottom-nav-item">
            <i class="fas fa-cog bottom-nav-icon"></i>
            <span class="bottom-nav-text">Settings</span>
        </a>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Show success dialog if needed
        const successDialogOverlay = document.getElementById('successDialogOverlay');
        const successCloseBtn = document.getElementById('successCloseBtn');
        
        <?php if ($showSuccessDialog): ?>
        successDialogOverlay.classList.add('active');
        <?php endif; ?>
        
        if (successCloseBtn) {
            successCloseBtn.addEventListener('click', function() {
                successDialogOverlay.classList.remove('active');
            });
        }
        
        // Close dialog when clicking outside
        if (successDialogOverlay) {
            successDialogOverlay.addEventListener('click', function(e) {
                if (e.target === successDialogOverlay) {
                    successDialogOverlay.classList.remove('active');
                }
            });
        }
        
        // Account dropdown functionality
        const accountBtn = document.getElementById('accountBtn');
        const accountDropdown = document.getElementById('accountDropdown');
        
        if (accountBtn && accountDropdown) {
            accountBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                accountDropdown.classList.toggle('active');
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', function() {
                accountDropdown.classList.remove('active');
            });
            
            // Prevent dropdown from closing when clicking inside it
            accountDropdown.addEventListener('click', function(e) {
                e.stopPropagation();
            });
        }
        
        // Right sidebar functionality
        const hamburgerBtn = document.getElementById('hamburgerBtn');
        const rightSidebar = document.getElementById('rightSidebar');
        const sidebarClose = document.getElementById('sidebarClose');
        const sidebarOverlay = document.getElementById('sidebarOverlay');
        
        if (hamburgerBtn && rightSidebar && sidebarClose && sidebarOverlay) {
            // Open sidebar when hamburger is clicked
            hamburgerBtn.addEventListener('click', function() {
                rightSidebar.classList.add('active');
                sidebarOverlay.classList.add('active');
                document.body.style.overflow = 'hidden'; // Prevent body scroll when sidebar is open
                
                // Reset animations when opening
                const menuItems = rightSidebar.querySelectorAll('.nk-menu-item');
                menuItems.forEach(item => {
                    item.style.animation = 'none';
                    item.offsetHeight; // Trigger reflow
                    item.style.animation = null;
                });
            });
            
            // Close sidebar when close button is clicked
            sidebarClose.addEventListener('click', function() {
                rightSidebar.classList.remove('active');
                sidebarOverlay.classList.remove('active');
                document.body.style.overflow = ''; // Restore body scroll
            });
            
            // Close sidebar when overlay is clicked
            sidebarOverlay.addEventListener('click', function() {
                rightSidebar.classList.remove('active');
                sidebarOverlay.classList.remove('active');
                document.body.style.overflow = ''; // Restore body scroll
            });
            
            // Handle submenu toggles
            const menuToggles = document.querySelectorAll('.nk-menu-toggle');
            menuToggles.forEach(toggle => {
                toggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    const parentItem = this.parentElement;
                    const submenu = parentItem.querySelector('.nk-menu-sub');
                    
                    if (submenu) {
                        submenu.classList.toggle('active');
                        this.classList.toggle('active');
                    }
                });
            });
        }
        
        // Preset amount buttons functionality
        const presetAmountBtns = document.querySelectorAll('.preset-amount-btn');
        const amountInput = document.getElementById('amountInput');
        
        if (presetAmountBtns && amountInput) {
            presetAmountBtns.forEach(button => {
                button.addEventListener('click', function() {
                    const amount = this.getAttribute('data-amount');
                    amountInput.value = amount;
                    
                    // Update active state
                    presetAmountBtns.forEach(btn => btn.classList.remove('active'));
                    this.classList.add('active');
                });
            });
        }
        
        // Form validation
        const withdrawalForm = document.getElementById('withdrawalForm');
        
        if (withdrawalForm) {
            withdrawalForm.addEventListener('submit', function(e) {
                const amountInput = document.querySelector('input[name="amount"]');
                const amount = parseFloat(amountInput.value);
                const minAmount = parseFloat(amountInput.getAttribute('min'));
                const maxAmount = parseFloat(amountInput.getAttribute('max'));
                
                if (!amount || amount < minAmount || amount > maxAmount) {
                    e.preventDefault();
                    
                    // Create error notification
                    const notification = document.createElement('div');
                    notification.className = 'copy-notification';
                    notification.style.backgroundColor = 'var(--danger)';
                    notification.textContent = 'Please select a valid amount before submitting';
                    
                    // Add the notification to the document
                    document.body.appendChild(notification);
                    
                    // Remove notification after 3 seconds
                    setTimeout(() => {
                        notification.style.opacity = '0';
                        setTimeout(() => {
                            if (notification.parentNode) {
                                notification.parentNode.removeChild(notification);
                            }
                        }, 500);
                    }, 3000);
                }
            });
        }
    });
    </script>
</body>
</html>